"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cognito_1 = require("monocdk/aws-cognito");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * Cognito user pools have password policies that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cognito_1.CfnUserPool) {
        const policies = monocdk_1.Stack.of(node).resolve(node.policies);
        if (policies == undefined) {
            return false;
        }
        const passwordPolicy = monocdk_1.Stack.of(node).resolve(policies.passwordPolicy);
        if (passwordPolicy == undefined) {
            return false;
        }
        const minimumLength = nag_pack_1.resolveIfPrimitive(node, passwordPolicy.minimumLength);
        if (minimumLength == undefined || minimumLength < 8) {
            return false;
        }
        const requireUppercase = nag_pack_1.resolveIfPrimitive(node, passwordPolicy.requireUppercase);
        if (requireUppercase !== true) {
            return false;
        }
        const requireNumbers = nag_pack_1.resolveIfPrimitive(node, passwordPolicy.requireNumbers);
        if (requireNumbers !== true) {
            return false;
        }
        const requireSymbols = nag_pack_1.resolveIfPrimitive(node, passwordPolicy.requireSymbols);
        if (requireSymbols !== true) {
            return false;
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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