"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_apigateway_1 = require("monocdk/aws-apigateway");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * Rest APIs have request validation enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_apigateway_1.CfnRestApi) {
        const apiLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_apigateway_1.CfnRequestValidator) {
                if (isMatchingRequestValidator(child, apiLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Request Validator is associated with the given Rest API
 * @param node the CfnRequestValidator to check
 * @param apiLogicalId the Cfn Logical ID of the REST API
 * returns whether the CfnRequestValidator is associated with the given Rest API
 */
function isMatchingRequestValidator(node, apiLogicalId) {
    const resourceLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.restApiId);
    if (resourceLogicalId === apiLogicalId) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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