"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_apigateway_1 = require("monocdk/aws-apigateway");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * Rest API stages are associated with AWS WAFv2 web ACLs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_apigateway_1.CfnStage) {
        const stageLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const stageName = nag_pack_1.resolveResourceFromInstrinsic(node, node.stageName);
        const restApiId = nag_pack_1.resolveResourceFromInstrinsic(node, node.restApiId);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                if (isMatchingWebACLAssociation(child, stageLogicalId, stageName, restApiId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Rest API
 * @param node the CfnWebACLAssociation to check
 * @param stageLogicalId the Cfn Logical ID of the REST API Stage
 * @param stageName the name of the REST API Stage
 * @param restApiId the ID or Cfn Resource ID of the REST API associated with the Stage
 * returns whether the CfnWebACLAssociation is associates with the given Rest API
 */
function isMatchingWebACLAssociation(node, stageLogicalId, stageName, restApiId) {
    const resourceLogicalId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    const regexes = Array();
    regexes.push(`${restApiId}.+${stageLogicalId}(?![\\w])`);
    if (stageName !== undefined) {
        regexes.push(`${restApiId}.+${stageName}(?![\\w])`);
    }
    const regex = new RegExp(regexes.join('|'), 'gm');
    if (regex.test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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