"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveResourceFromInstrinsic = exports.resolveIfPrimitive = exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 *
 * @stability stable
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel[NagMessageLevel["WARN"] = 0] = "WARN";
    NagMessageLevel[NagMessageLevel["ERROR"] = 1] = "ERROR";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 *
 * @stability stable
 */
class NagPack {
    /**
     * @stability stable
     */
    constructor(props) {
        /**
         * @stability stable
         */
        this.packName = '';
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
    }
    /**
     * @stability stable
     */
    get readPackName() {
        return this.packName;
    }
    /**
     * Create a rule to be used in the NagPack.
     *
     * @param params The.
     * @stability stable
     * @IApplyRule interface with rule details.
     */
    applyRule(params) {
        var _b, _c, _d;
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        let resourceIgnores = (_b = params.node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
        resourceIgnores = resourceIgnores ? resourceIgnores : [];
        let stackIgnores = (_d = (_c = monocdk_1.Stack.of(params.node).templateOptions.metadata) === null || _c === void 0 ? void 0 : _c.cdk_nag) === null || _d === void 0 ? void 0 : _d.rules_to_suppress;
        stackIgnores = stackIgnores ? stackIgnores : [];
        const allIgnores = resourceIgnores.concat(stackIgnores);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        try {
            if (!params.rule(params.node)) {
                const reason = this.ignoreRule(allIgnores, ruleId);
                if (reason) {
                    if (this.logIgnores === true) {
                        const message = this.createMessage(SUPPRESSION_ID, `${ruleId} was triggered but suppressed.`, `Provided reason: "${reason}"`);
                        monocdk_1.Annotations.of(params.node).addInfo(message);
                    }
                }
                else {
                    const message = this.createMessage(ruleId, params.info, params.explanation);
                    if (params.level == NagMessageLevel.ERROR) {
                        monocdk_1.Annotations.of(params.node).addError(message);
                    }
                    else if (params.level == NagMessageLevel.WARN) {
                        monocdk_1.Annotations.of(params.node).addWarning(message);
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID);
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    monocdk_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(ignores, ruleId) {
        for (let ignore of ignores) {
            if (ignore.id &&
                ignore.reason &&
                JSON.stringify(ignore.reason).length >= 10) {
                if (ignore.id == ruleId) {
                    return ignore.reason;
                }
            }
            else {
                throw Error(`Improperly formatted cdk_nag rule suppression detected: ${JSON.stringify(ignore)}. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    createMessage(ruleId, info, explanation) {
        let message = `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "1.0.3" };
/**
 * Use in cases where a primitive value must be known to pass a rule.
 * https://developer.mozilla.org/en-US/docs/Glossary/Primitive
 * @param node The CfnResource to check.
 * @param parameter The value to attempt to resolve.
 * @returns Return a value if resolves to a primitive data type, otherwise throw an error.
 */
function resolveIfPrimitive(node, parameter) {
    const resolvedValue = monocdk_1.Stack.of(node).resolve(parameter);
    if (resolvedValue === Object(resolvedValue)) {
        throw Error(`The parameter resolved to to a non-primitive value "${JSON.stringify(resolvedValue)}", therefore the rule could not be validated.`);
    }
    else {
        return resolvedValue;
    }
}
exports.resolveIfPrimitive = resolveIfPrimitive;
/**
 * Use in cases where a token resolves to an intrinsic function and the referenced resource must be known to pass a rule.
 * @param node The CfnResource to check.
 * @param parameter The value to attempt to resolve.
 * @returns Return the Logical resource Id if resolves to a intrinsic function, otherwise the resolved provided value.
 */
function resolveResourceFromInstrinsic(node, parameter) {
    const resolvedValue = monocdk_1.Stack.of(node).resolve(parameter);
    const ref = resolvedValue === null || resolvedValue === void 0 ? void 0 : resolvedValue.Ref;
    const getAtt = resolvedValue === null || resolvedValue === void 0 ? void 0 : resolvedValue['Fn::GetAtt'];
    if (ref != undefined) {
        return ref;
    }
    else if (Array.isArray(getAtt) && getAtt.length > 0) {
        return getAtt[0];
    }
    return resolvedValue;
}
exports.resolveResourceFromInstrinsic = resolveResourceFromInstrinsic;
//# sourceMappingURL=data:application/json;base64,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