# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['linguee_api', 'linguee_api.downloaders', 'tests', 'tests.parsers']

package_data = \
{'': ['*']}

install_requires = \
['fastapi>=0.65.2,<0.66.0',
 'httpx==1.0.0b0',
 'loguru>=0.5.3,<0.6.0',
 'pydantic>=1.8.1,<2.0.0',
 'python-dotenv>=0.17.0,<0.18.0',
 'sentry-sdk>=1.0.0,<2.0.0',
 'uvicorn>=0.13.4,<0.14.0',
 'xextract>=0.1.8,<0.2.0']

setup_kwargs = {
    'name': 'linguee-api',
    'version': '2.4.0',
    'description': 'Linguee API',
    'long_description': '# Linguee API\n\n[Linguee](https://linguee.com) provides excellent dictionary and translation memory service. Unfortunately, there is no way you can get automated access to it. Linguee API fixes the problem. It acts as a proxy and converts their HTML responses to easy-to-use JSON API.\n\n## API endpoints\n\nThe proxy provides three API endpoints: for translations, for examples, and external sources.\n\n![Linguee API](./docs/linguee-api.png)\n\nThe API documentation and the playground is available for the sample installation:\n\n- [Documentation and API playground](https://linguee-api-v2.herokuapp.com/docs)\n- [The same documentation, but formatted with ReDoc](https://linguee-api-v2.herokuapp.com/redoc)\n\n## Sample installation\n\nSample installation is available at https://linguee-api-v2.herokuapp.com.\n\n- Get translations of the word "bacalhau" from Portuguese to English: [https://linguee-api-v2.herokuapp.com/api/v2/translations?query=bacalhau&src=pt&dst=en](https://linguee-api-v2.herokuapp.com/api/v2/translations?query=bacalhau&src=pt&dst=en).\n- Get a list of curated examples: [https://linguee-api-v2.herokuapp.com/api/v2/examples?query=bacalhau&src=pt&dst=en](https://linguee-api-v2.herokuapp.com/api/v2/examples?query=bacalhau&src=pt&dst=en).\n- Get examples from external sources: [https://linguee-api-v2.herokuapp.com/api/v2/external_sources?query=bacalhau&src=pt&dst=en](https://linguee-api-v2.herokuapp.com/api/v2/examples?query=bacalhau&src=pt&dst=en).\n\n[![Deploy](https://www.herokucdn.com/deploy/button.svg)](https://heroku.com/deploy)\n\n## Local installation\n\nInstall the Linguee API.\n\n```shell\n$ pip install linguee-api\n```\n\nRun the API server with `uvicorn` (installed as a dependency.)\n\n```shell\n$ uvicorn linguee_api.api:app\n...\nINFO:     Uvicorn running on http://127.0.0.1:8000 (Press CTRL+C to quit)\n...\n```\n\nOpen http://127.0.0.1:8000. You will be redirected to the API documentation page, where you can test the API.\n\n## Supported languages\n\nAPI supports all the languages, supported by Linguee. As in Linguee, not all language pairs are valid though. Supported languages:\n`bg` (Bulgarian), `cs` (Czech), `da` (Danish), `de` (German), `el` (Greek), `en` (English), `es` (Spanish), `et` (Estonian), `fi` (Finnish), `fr` (French), `hu` (Hungarian), `it` (Italian), `ja` (Japan),`lt` (Lithuanian), `lv` (Latvian), `mt` (Maltese), `nl` (Dutch), `pl` (Polish), `pt` (Portuguese), `ro` (Romanian), `ru` (Russian), `sk` (Slovak), `sl` (Solvene), `sv` (Swedish), `zh` (Chinese).\n\n## Response structure\n\n**Lemmas**\n\nEvery query (a random string) can match several so-called lemma objects.\n\nAccording to Wikipedia, [lemma](https://en.wikipedia.org/wiki/Lemma_(morphology)) is the canonical form, dictionary form, or citation form of a set of words.\n\nIn English, for example, break, breaks, broke, broken, and breaking are forms of the same lexeme, with "break" as the lemma by which they are indexed.\n\nIn the API, lemmas have the only required attribute, "text," but may have optional elements, such as part of speech ("pos") and audio links with pronunciations.\n\n\n**Translations**\n\nEvery lemma has one or more translations. The translation is a lemma in a different language and has a similar structure with the necessary text field and optional part of speech and audio links.\n\n\n**Examples**\n\nIn addition to lemmas, the API returns several usage examples curated by dictionary authors. Examples are the short phrases, annotated with one or more equivalents in different languages. When appropriate, examples may contain the part-of-speech form and audio links.\n\n**External Sources**\n\nOn top of curated examples, Linguee provides links to external sources. The API returns objects containing the phrase snipped in the original language and an equivalent snippet in the translation.\n\n## Usage examples with Python and requests\n\nOnce installed on Heroku, Linguee API can be used as any other API service. I recommend using the [requests](https://docs.python-requests.org/) library.\n\n### Translate a word or a phrase from one language to another with Python\n\nA request to the sample API installation to translate the word "bacalhau" from Portuguese to English.\n\n```python\nimport requests\n\napi_root = "https://linguee-api-v2.herokuapp.com/api/v2"\nresp = requests.get(f"{api_root}/translations", params={"query": "bacalhau", "src": "pt", "dst": "en"})\nfor lemma in resp.json():\n    for translation in lemma[\'translations\']:\n        print(f"{lemma[\'text\']} -> {translation[\'text\']}")\n```\n\nThis will print:\n\n```\nbacalhau -> cod\nbacalhau -> codfish\n```\n\n### Provide translation examples with Python\n\nA request to the sample API installation to get all usage examples of "bacalhau" along with their translations.\n\n```python\nimport requests\n\napi_root = "https://linguee-api-v2.herokuapp.com/api/v2"\n\nresp = requests.get(f"{api_root}/examples", params={"query": "bacalhau", "src": "pt", "dst": "en"})\n\nfor example in resp.json():\n    for translation in example["translations"]:\n        print(f"{example[\'text\']} -> {translation[\'text\']}")\n```\n\nThis will print:\n\n```\nbacalhau desfiado -> shredded cod\nlombo de bacalhau -> codfish fillet\n...\nbacalhau do Atlântico -> Atlantic cod\n```\n\n### Get access to real world usage examples with Python\n\nA request to the sample API installation to get all real-world usage examples of "bacalhau" along with their translations.\n\n```python\nimport requests\n\napi_root = "https://linguee-api-v2.herokuapp.com/api/v2"\n\nresp = requests.get(f"{api_root}/external_sources", params={"query": "bacalhau", "src": "pt", "dst": "en"})\nfor source in resp.json():\n    print(f"{source[\'src\']} -> {source[\'dst\']}")\n```\n\nThis will print a long list of real-world examples like this:\n\n```\nÉ calculado o esforço de [...] pesca de todos os navios que capturam bacalhau. -> The fishing effort of all [...] the vessels catching cod will be calculated.\n```\n\n\n## Bash, curl and jq usage example\n\nOnce installed on Heroku, Linguee API can be used as any other API service.\n\nFor Bash scripts you can use curl and [jq](https://stedolan.github.io/jq/), a command-line JSON parser.\n\n### Translate a word or a phrase from one language to another with Bash\n\nA request to the sample API installation to get all usage examples of "bacalhau" along with their translations.\n\n```bash\ncurl -s \'https://linguee-api-v2.herokuapp.com/api/v2/translations?query=bacalhau&src=pt&dst=en\' | jq -c \'{text: .[].text, translation: .[].translations[].text}\'\n```\n\nThis will print\n\n```json lines\n{"text":"bacalhau","translation":"cod"}\n{"text":"bacalhau","translation":"codfish"}\n```\n\n### Provide translation examples with Bash\n\nA request to the sample API installation to get all usage examples of "bacalhau" along with their translations.\n\n```shell\ncurl -s \'https://linguee-api-v2.herokuapp.com/api/v2/examples?query=bacalhau&src=pt&dst=en\' | jq -c \'{text: .[].text, translation: .[].translations[].text}\'\n```\n\nThis will print something like this:\n\n```json lines\n{"text":"bacalhau desfiado","translation":"shredded cod"}\n{"text":"bacalhau desfiado","translation":"codfish fillet"}\n...\n{"text":"bacalhau do Atlântico","translation":"Atlantic cod"}\n```\n\n### Get access to real world usage examples with Bash\n\nA request to the sample API installation to get all real-world usage examples of "bacalhau" along with their translations.\n\n```shell\ncurl -s \'https://linguee-api-v2.herokuapp.com/api/v2/external_sources?query=bacalhau&src=pt&dst=en\' | jq -c \'{src: .[].src, dst: .[].dst}\'\n```\n\nThis will print a long list of real-world examples like this:\n\n```json lines\n{"src":"É calculado o esforço de [...] pesca de todos os navios que capturam bacalhau.","dst":"The fishing effort of all [...] the vessels catching cod will be calculated."}\n...\n```\n\n## Terms and Conditions\n\nIf you use the API, make sure you comply with\n[Linguee Terms and Conditions](http://www.linguee.com/page/termsAndConditions.php),\nand in particular with that clause:\n\n> Both private and business usage of linguee.com services is free of charge.\n> It is however strictly prohibited to forward on our services to third\n> parties against payment\n',
    'author': 'Roman Imankulov',
    'author_email': 'roman.imankulov@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/imankulov/linguee-api',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
