"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const association_manager_1 = require("./private/association-manager");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const servicecatalog_generated_1 = require("./servicecatalog.generated");
class PortfolioBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
    }
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    addProduct(product) {
        association_manager_1.AssociationManager.associateProductWithPortfolio(this, product, undefined);
    }
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        new servicecatalog_generated_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.messageLanguage,
        });
    }
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, this.portfolioId, tagOptions);
    }
    constrainTagUpdates(product, options = {}) {
        association_manager_1.AssociationManager.constrainTagUpdates(this, product, options);
    }
    notifyOnStackEvents(product, topic, options = {}) {
        association_manager_1.AssociationManager.notifyOnStackEvents(this, product, topic, options);
    }
    constrainCloudFormationParameters(product, options) {
        association_manager_1.AssociationManager.constrainCloudFormationParameters(this, product, options);
    }
    setLaunchRole(product, launchRole, options = {}) {
        association_manager_1.AssociationManager.setLaunchRole(this, product, launchRole, options);
    }
    setLocalLaunchRoleName(product, launchRoleName, options = {}) {
        const launchRole = new iam.Role(this, `LaunchRole${launchRoleName}`, {
            roleName: launchRoleName,
            assumedBy: new iam.ServicePrincipal('servicecatalog.amazonaws.com'),
        });
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
        return launchRole;
    }
    setLocalLaunchRole(product, launchRole, options = {}) {
        validation_1.InputValidator.validateRoleNameSetForLocalLaunchRole(launchRole);
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
    }
    deployWithStackSets(product, options) {
        association_manager_1.AssociationManager.deployWithStackSets(this, product, options);
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new servicecatalog_generated_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
}
/**
 * A Service Catalog portfolio.
 */
class Portfolio extends PortfolioBase {
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_servicecatalog_PortfolioProps(props);
        this.validatePortfolioProps(props);
        this.portfolio = new servicecatalog_generated_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.messageLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = cdk.Stack.of(this).formatArn({
            service: 'servicecatalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    /**
     * Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = cdk.Stack.of(scope).splitArn(portfolioArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return util_1.hashValues(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    generateUniqueHash(value) {
        return util_1.hashValues(cdk.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
exports.Portfolio = Portfolio;
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "@aws-cdk/aws-servicecatalog.Portfolio", version: "1.151.0" };
//# sourceMappingURL=data:application/json;base64,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