from typing import Any, Dict, List, Optional, Union

import httpx

from ...client import AuthenticatedClient
from ...exceptions import BadRequestError, HTTPClientError, ParamValidationError, UnauthorizedTokenError
from ...models.http_validation_error import HTTPValidationError
from ...models.message_model import MessageModel
from ...models.repo_all_response import RepoAllResponse
from ...types import UNSET, Response, Unset


def _get_kwargs(
    *,
    client: AuthenticatedClient,
    repo_owner: str,
    repo_name: str,
    oblivious_user_id: str,
    account_type: Union[Unset, None, str] = "github",
) -> Dict[str, Any]:
    url = "{}/repo".format(client.base_url)

    headers: Dict[str, str] = client.get_headers()
    cookies: Dict[str, Any] = client.get_cookies()

    params: Dict[str, Any] = {}
    params["repo_owner"] = repo_owner

    params["repo_name"] = repo_name

    params["oblivious_user_id"] = oblivious_user_id

    params["account_type"] = account_type

    params = {k: v for k, v in params.items() if v is not UNSET and v is not None}

    return {
        "method": "post",
        "url": url,
        "headers": headers,
        "cookies": cookies,
        "timeout": client.get_timeout(),
        "params": params,
    }


def _parse_response(
    *, response: httpx.Response
) -> Optional[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]:
    if response.status_code == 200:
        response_200 = []
        _response_200 = response.json()
        for response_200_item_data in _response_200:
            response_200_item = RepoAllResponse.from_dict(response_200_item_data)

            response_200.append(response_200_item)

        return response_200
    if response.status_code == 400:
        response_400_message = response.json()["message"]
        raise BadRequestError(message=response_400_message)
    if response.status_code == 500:
        response_500_request_id = response.headers["apigw-requestid"]
        raise HTTPClientError(request_id=response_500_request_id)
    if response.status_code == 422:
        response_422 = HTTPValidationError.from_dict(response.json())
        if response_422.detail[0].type.__contains__("regex"):
            report = "Invalid " + response_422.detail[0].loc[-1] + " provided"
        report = "Invalid " + response_422.detail[0].loc[-1] + " provided"
        raise ParamValidationError(report=report)
    if response.status_code == 403:
        raise UnauthorizedTokenError()
    return None


def _build_response(
    *, response: httpx.Response
) -> Response[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]:
    return Response(
        status_code=response.status_code,
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(response=response),
    )


def sync_detailed(
    *,
    client: AuthenticatedClient,
    repo_owner: str,
    repo_name: str,
    oblivious_user_id: str,
    account_type: Union[Unset, None, str] = "github",
) -> Response[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]:
    """Link Repo

     API to link a repo to user.

    Args:
        repo_owner (str):
        repo_name (str):
        oblivious_user_id (str):
        account_type (Union[Unset, None, str]):  Default: 'github'.

    Returns:
        Response[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]
    """

    kwargs = _get_kwargs(
        client=client,
        repo_owner=repo_owner,
        repo_name=repo_name,
        oblivious_user_id=oblivious_user_id,
        account_type=account_type,
    )

    response = httpx.request(
        verify=client.verify_ssl,
        **kwargs,
    )

    return _build_response(response=response)


def sync(
    *,
    client: AuthenticatedClient,
    repo_owner: str,
    repo_name: str,
    oblivious_user_id: str,
    account_type: Union[Unset, None, str] = "github",
) -> Optional[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]:
    """Link Repo

     API to link a repo to user.

    Args:
        repo_owner (str):
        repo_name (str):
        oblivious_user_id (str):
        account_type (Union[Unset, None, str]):  Default: 'github'.

    Returns:
        Response[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]
    """

    return sync_detailed(
        client=client,
        repo_owner=repo_owner,
        repo_name=repo_name,
        oblivious_user_id=oblivious_user_id,
        account_type=account_type,
    ).parsed


async def asyncio_detailed(
    *,
    client: AuthenticatedClient,
    repo_owner: str,
    repo_name: str,
    oblivious_user_id: str,
    account_type: Union[Unset, None, str] = "github",
) -> Response[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]:
    """Link Repo

     API to link a repo to user.

    Args:
        repo_owner (str):
        repo_name (str):
        oblivious_user_id (str):
        account_type (Union[Unset, None, str]):  Default: 'github'.

    Returns:
        Response[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]
    """

    kwargs = _get_kwargs(
        client=client,
        repo_owner=repo_owner,
        repo_name=repo_name,
        oblivious_user_id=oblivious_user_id,
        account_type=account_type,
    )

    async with httpx.AsyncClient(verify=client.verify_ssl) as _client:
        response = await _client.request(**kwargs)

    return _build_response(response=response)


async def asyncio(
    *,
    client: AuthenticatedClient,
    repo_owner: str,
    repo_name: str,
    oblivious_user_id: str,
    account_type: Union[Unset, None, str] = "github",
) -> Optional[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]:
    """Link Repo

     API to link a repo to user.

    Args:
        repo_owner (str):
        repo_name (str):
        oblivious_user_id (str):
        account_type (Union[Unset, None, str]):  Default: 'github'.

    Returns:
        Response[Union[Any, HTTPValidationError, List[RepoAllResponse], MessageModel]]
    """

    return (
        await asyncio_detailed(
            client=client,
            repo_owner=repo_owner,
            repo_name=repo_name,
            oblivious_user_id=oblivious_user_id,
            account_type=account_type,
        )
    ).parsed
