"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = void 0;
const core_1 = require("@aws-cdk/core");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * A certificate managed by AWS Certificate Manager
 *
 * IMPORTANT: if you are creating a certificate as part of your stack, the stack
 * will not complete creating until you read and follow the instructions in the
 * email that you will receive.
 *
 * ACM will send validation emails to the following addresses:
 *
 *  admin@domain.com
 *  administrator@domain.com
 *  hostmaster@domain.com
 *  postmaster@domain.com
 *  webmaster@domain.com
 *
 * For every domain that you register.
 */
class Certificate extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: allDomainNames.map(domainValidationOption),
            validationMethod: props.validationMethod,
        });
        this.certificateArn = cert.ref;
        /**
         * Return the domain validation options for the given domain
         *
         * Closes over props.
         */
        function domainValidationOption(domainName) {
            let validationDomain = props.validationDomains && props.validationDomains[domainName];
            if (validationDomain === undefined) {
                if (core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                validationDomain = util_1.apexDomain(domainName);
            }
            return { domainName, validationDomain };
        }
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
//# sourceMappingURL=data:application/json;base64,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