#!/usr/bin/env python
# vim: set fileencoding=utf-8 :


"""Tests audio reader and writer based on sox for bob and python
"""

import nose.tools

import scipy.io.wavfile
import numpy

import bob.io.base
from . import reader, writer

from bob.io.base.test_utils import datafile, temporary_filename


def F(f):
  """Returns the test file on the "data" subdirectory"""
  return datafile(f, __name__)


def run_wav(fname):

  # audio file name
  infile = F(fname)

  # reading the file using our SoX bindings
  f = reader(infile)
  data = f.load()

  # check bob.io.base interface
  assert numpy.array_equal(data, bob.io.base.load(infile))

  # reading the file using scipy.io.wavfile
  (scipy_rate, scipy_data) = scipy.io.wavfile.read(infile)

  # verify if the rates are the same
  nose.tools.eq_(int(f.rate), scipy_rate)

  # If one channel
  if data.shape[0] == 1:
    # verify if the data is the same (x 2^15)
    assert numpy.array_equal((data[0]*pow(2,15)).astype(int), scipy_data)

  # If more than one channel
  elif data.shape[0] > 1:

    # verify if the number of channels is the same
    nose.tools.eq_(data.shape[0], scipy_data.shape[1])

    # verify if the data are the same (x 2^15)
    for i in range(data.shape[0]):
      assert numpy.array_equal((data[i]*pow(2,15)).astype(int), scipy_data[:,i])


def test_read_mono_wav():
  run_wav('sample1.wav')


def test_read_stereo_wav():
  run_wav('sample4.wav')


def test_compare_wav_matlab():

  # audio file name
  infile = F('sample3.wav')

  # reading the file using xbob.sox
  f = reader(infile)
  data = f.load()

  # reading the .mat file that contains the data generated by the Matlab
  # wavread function
  mat_data = bob.io.base.load(F('sample3_matlab.hdf5'))
  assert numpy.array_equal(data, mat_data), '%r != %r' % (data, mat_data)


def run_check(stem, fmt1, fmt2):

  f1 = reader(F(stem + fmt1))
  f2 = reader(F(stem + fmt2))

  nose.tools.eq_(f1.rate, f2.rate)

  data_f1 = f1.load()
  data_f2 = f2.load()

  # verify the data is the same
  assert numpy.array_equal(data_f1, data_f2), '%r != %r' % (data_f1, data_f2)


def test_sph_timit_1():
  run_check('timit-sa1', '.wav', '.sph')

def test_sph_timit_2():
  run_check('timit-sa2', '.wav', '.sph')


def read_write(stem, fmt1, fmt2, encoding='UNKNOWN', bits_per_sample=16):

  f1 = reader(F(stem + fmt1))
  data = f1.load()

  f2_filename = temporary_filename(suffix=fmt2)
  f2 = writer(f2_filename, rate=f1.rate, encoding=encoding,
      bits_per_sample=bits_per_sample)
  f2.append(data)
  f2.close() #forces file closing

  f2 = reader(f2_filename)

  nose.tools.eq_(f1.rate, f2.rate)

  data_f1 = f1.load()
  data_f2 = f2.load()

  # verify the data is the same
  assert numpy.array_equal(data_f1, data_f2), '%r != %r' % (data_f1, data_f2)


def test_sph_writing_timit_1():
  read_write('timit-sa1', '.wav', '.sph')

def test_sph_writing_timit_2():
  read_write('timit-sa2', '.wav', '.sph')

def test_wav_writing_timit_1():
  read_write('timit-sa1', '.sph', '.wav')

def test_wav_writing_timit_2():
  read_write('timit-sa2', '.sph', '.wav')

def test_au_writing_timit_1():
  read_write('timit-sa1', '.sph', '.au')

def test_au_writing_timit_2():
  read_write('timit-sa2', '.sph', '.au')
