"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const props = {
        deployLambda: true,
        lambdaFunctionProps: {
            code: lambda.Code.asset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
    };
    return new lib_1.DynamoDBStreamToLambda(stack, 'test-lambda-dynamodb-stack', props);
}
test('snapshot test DynamoDBStreamToLambda default params', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check lambda EventSourceMapping', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Lambda::EventSourceMapping', {
        EventSourceArn: {
            "Fn::GetAtt": [
                "testlambdadynamodbstackDynamoTable8138E93B",
                "StreamArn"
            ]
        },
        FunctionName: {
            Ref: "LambdaFunctionBF21E41F"
        },
        BatchSize: 100,
        StartingPosition: "TRIM_HORIZON"
    });
});
test('check DynamoEventSourceProps override', () => {
    const stack = new cdk.Stack();
    const props = {
        deployLambda: true,
        lambdaFunctionProps: {
            code: lambda.Code.asset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        dynamoEventSourceProps: {
            startingPosition: lambda.StartingPosition.LATEST,
            batchSize: 55
        }
    };
    new lib_1.DynamoDBStreamToLambda(stack, 'test-lambda-dynamodb-stack', props);
    expect(stack).toHaveResource('AWS::Lambda::EventSourceMapping', {
        EventSourceArn: {
            "Fn::GetAtt": [
                "testlambdadynamodbstackDynamoTable8138E93B",
                "StreamArn"
            ]
        },
        FunctionName: {
            Ref: "LambdaFunctionBF21E41F"
        },
        BatchSize: 55,
        StartingPosition: "LATEST"
    });
});
test('check lambda permission to read dynamodb stream', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:ListStreams",
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::GetAtt": [
                                        "testlambdadynamodbstackDynamoTable8138E93B",
                                        "Arn"
                                    ]
                                },
                                "/stream/*"
                            ]
                        ]
                    }
                },
                {
                    Action: [
                        "dynamodb:DescribeStream",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testlambdadynamodbstackDynamoTable8138E93B",
                            "StreamArn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "LambdaFunctionServiceRoleDefaultPolicy126C8897",
        Roles: [
            {
                Ref: "LambdaFunctionServiceRole0C4CDE0B"
            }
        ]
    });
});
test('check dynamodb table stream override', () => {
    const stack = new cdk.Stack();
    const props = {
        deployLambda: true,
        lambdaFunctionProps: {
            code: lambda.Code.asset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        dynamoTableProps: {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        }
    };
    new lib_1.DynamoDBStreamToLambda(stack, 'test-lambda-dynamodb-stack', props);
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        },
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    });
});
test('check getter methods', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.lambdaFunction()).toBeInstanceOf(lambda.Function);
    expect(construct.dynamoTable()).toBeInstanceOf(dynamodb.Table);
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        deployLambda: true
    };
    try {
        new lib_1.DynamoDBStreamToLambda(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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