"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const defaults = require("@aws-solutions-konstruk/core");
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-konstruk/core");
class DynamoDBStreamToLambda extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {DynamoDBStreamToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.fn = defaults.buildLambdaFunction(scope, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Set the default props for DynamoDB table
        if (props.dynamoTableProps) {
            const dynamoTableProps = core_2.overrideProps(defaults.DefaultTableWithStreamProps, props.dynamoTableProps);
            this.table = new dynamodb.Table(this, 'DynamoTable', dynamoTableProps);
        }
        else {
            this.table = new dynamodb.Table(this, 'DynamoTable', defaults.DefaultTableWithStreamProps);
        }
        // Grant DynamoDB Stream read perimssion for lambda function
        this.table.grantStreamRead(this.fn.grantPrincipal);
        // Create DynamDB trigger to invoke lambda function
        this.fn.addEventSource(new aws_lambda_event_sources_1.DynamoEventSource(this.table, defaults.DynamoEventSourceProps(props.dynamoEventSourceProps)));
    }
    /**
     * @summary Retruns an instance of dynamodb.Table created by the construct.
     * @returns {dynamodb.Table} Instance of dynamodb.Table created by the construct
     * @since 0.8.0
     * @access public
     */
    dynamoTable() {
        return this.table;
    }
    /**
     * @summary Retruns an instance of lambda.Function created by the construct.
     * @returns {lambda.Function} Instance of lambda.Function created by the construct
     * @since 0.8.0
     * @access public
     */
    lambdaFunction() {
        return this.fn;
    }
}
exports.DynamoDBStreamToLambda = DynamoDBStreamToLambda;
//# sourceMappingURL=data:application/json;base64,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