# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['multisort']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'multisort',
    'version': '0.1.1',
    'description': 'NoneType Safe Multi Column Sorting For Python',
    'long_description': '## `multisort` - NoneType Safe Multi Column Sorting For Python\n\nSimplified multi-column sorting of lists of tuples, dicts, lists or objects that are NoneType safe.\n\n### Installation\n\n```\npython3 -m pip install multisort\n```\n\n### Dependencies\nNone\n\n### Performance\nAverage over 10 iterations with 500 rows.\nTest | Secs\n---|---\ncmp_func|0.0054\npandas|0.0061\nreversor|0.0149\nmsorted|0.0179\n\nAs you can see, if the `cmp_func` is by far the fastest methodology as long as the number of cells in the table are 500 rows for 5 columns. However for larger data sets, `pandas` is the performance winner and scales extremely well. In such large dataset cases, where performance is key, `pandas` should be the first choice.\n\nThe surprising thing from testing is that `cmp_func` far outperforms `reversor` which which is the only other methodology for multi-columnar sorting that can handle `NoneType` values.\n\n### Note on `NoneType` and sorting\nIf your data may contain None, it would be wise to ensure your sort algorithm is tuned to handle them. This is because sorted uses `<` comparisons; which is not supported by `NoneType`. For example, the following error will result: `TypeError: \'>\' not supported between instances of \'NoneType\' and \'str\'`.\n\n### Methodologies\nMethod|Descr|Notes\n---|---|---\ncmp_func|Multi column sorting in the model `java.util.Comparator`|Fastest for small to medium size data\nreversor|Enable multi column sorting with column specific reverse sorting|Medium speed. [Source](https://stackoverflow.com/a/56842689/286807)\nmsorted|Simple one-liner designed after `multisort` [example from python docs](https://docs.python.org/3/howto/sorting.html#sort-stability-and-complex-sorts)|Slowest of the bunch but not by much\n\n\n\n### Dictionary Examples\nFor data:\n```\nrows_dict = [\n     {\'idx\': 0, \'name\': \'joh\', \'grade\': \'C\', \'attend\': 100}\n    ,{\'idx\': 1, \'name\': \'jan\', \'grade\': \'a\', \'attend\': 80}\n    ,{\'idx\': 2, \'name\': \'dav\', \'grade\': \'B\', \'attend\': 85}\n    ,{\'idx\': 3, \'name\': \'bob\' , \'grade\': \'C\', \'attend\': 85}\n    ,{\'idx\': 4, \'name\': \'jim\' , \'grade\': \'F\', \'attend\': 55}\n    ,{\'idx\': 5, \'name\': \'joe\' , \'grade\': None, \'attend\': 55}\n]\n```\n\n### `msorted`\nSort rows_dict by _grade_, descending, then _attend_, ascending and put None first in results:\n```\nfrom multisort import msorted\nrows_sorted = msorted(rows_dict, [\n         (\'grade\', {\'reverse\': False, \'none_first\': True})\n        ,\'attend\'\n])\n\n```\n\nSort rows_dict by _grade_, descending, then _attend_ and call upper() for _grade_:\n```\nfrom multisort import msorted\nrows_sorted = msorted(rows_dict, [\n         (\'grade\', {\'reverse\': False, \'clean\': lambda s:None if s is None else s.upper()})\n        ,\'attend\'\n])\n\n```\n`msorted` parameters:\noption|dtype|description\n---|---|---\n`key`|int or str|Key to access data. int for tuple or list\n`spec`|str, int, list|Sort specification. Can be as simple as a column key / index\n`reverse`|bool|Reverse order of final sort (defalt = False)\n\n`msorted` `spec` options:\noption|dtype|description\n---|---|---\nreverse|bool|Reverse sort of column\nclean|func|Function / lambda to clean the value\nnone_first|bool|If True, None will be at top of sort. Default is False (bottom)\n\n\n\n### `sorted` with `reversor`\nSort rows_dict by _grade_, descending, then _attend_ and call upper() for _grade_:\n```\nrows_sorted = sorted(rows_dict, key=lambda o: (\n             reversor(None if o[\'grade\'] is None else o[\'grade\'].upper())\n            ,o[\'attend\'])\n))\n```\n\n\n### `sorted` with `cmp_func`\nSort rows_dict by _grade_, descending, then _attend_ and call upper() for _grade_:\n```\ndef cmp_student(a,b):\n    k=\'grade\'; va=a[k]; vb=b[k]\n    if va != vb: \n        if va is None: return -1\n        if vb is None: return 1\n        return -1 if va > vb else 1\n    k=\'attend\'; va=a[k]; vb=b[k]; \n    if va != vb: return -1 if va < vb else 1\n    return 0\nrows_sorted = sorted(rows_dict, key=cmp_func(cmp_student), reverse=True)\n```\n\n\n\n### Object Examples\nFor data:\n```\nclass Student():\n    def __init__(self, idx, name, grade, attend):\n        self.idx = idx\n        self.name = name\n        self.grade = grade\n        self.attend = attend\n    def __str__(self): return f"name: {self.name}, grade: {self.grade}, attend: {self.attend}"\n    def __repr__(self): return self.__str__()\n\nrows_obj = [\n     Student(0, \'joh\', \'C\', 100)\n    ,Student(1, \'jan\', \'a\', 80)\n    ,Student(2, \'dav\', \'B\', 85)\n    ,Student(3, \'bob\', \'C\', 85)\n    ,Student(4, \'jim\', \'F\', 55)\n    ,Student(5, \'joe\', None, 55)\n]\n```\n\n### `msorted`\n(Same syntax as with \'dict\' example)\n\n\n### `sorted` with `reversor`\nSort rows_obj by _grade_, descending, then _attend_ and call upper() for _grade_:\n```\nrows_sorted = sorted(rows_obj, key=lambda o: (\n             reversor(None if o.grade is None else o.grade.upper())\n            ,o.attend)\n))\n```\n\n\n### `sorted` with `cmp_func`\nSort rows_obj by _grade_, descending, then _attend_ and call upper() for _grade_:\n```\ndef cmp_student(a,b):\n    if a.grade != b.grade: \n        if a.grade is None: return -1\n        if b.grade is None: return 1\n        return -1 if a.grade > b.grade else 1\n    if a.attend != b.attend: \n        return -1 if a.attend < b.attend else 1\n    return 0\nrows_sorted = sorted(rows_obj, key=cmp_func(cmp_student), reverse=True)\n```\n\n\n### List / Tuple Examples\nFor data:\n```\nrows_tuple = [\n     (0, \'joh\', \'a\'  , 100)\n    ,(1, \'joe\', \'B\'  , 80)\n    ,(2, \'dav\', \'A\'  , 85)\n    ,(3, \'bob\', \'C\'  , 85)\n    ,(4, \'jim\', None , 55)\n    ,(5, \'jan\', \'B\'  , 70)\n]\n(COL_IDX, COL_NAME, COL_GRADE, COL_ATTEND) = range(0,4)\n```\n\n### `msorted`\nSort rows_tuple by _grade_, descending, then _attend_, ascending and put None first in results:\n```\nfrom multisort import msorted\nrows_sorted = msorted(rows_tuple, [\n         (COL_GRADE, {\'reverse\': False, \'none_first\': True})\n        ,COL_ATTEND\n])\n\n```\n\n\n### `sorted` with `reversor`\nSort rows_tuple by _grade_, descending, then _attend_ and call upper() for _grade_:\n```\nrows_sorted = sorted(rows_tuple, key=lambda o: (\n             reversor(None if o[COL_GRADE] is None else o[COL_GRADE].upper())\n            ,o[COL_ATTEND])\n))\n```\n\n\n### `sorted` with `cmp_func`\nSort rows_tuple by _grade_, descending, then _attend_ and call upper() for _grade_:\n```\ndef cmp_student(a,b):\n    k=COL_GRADE; va=a[k]; vb=b[k]\n    if va != vb: \n        if va is None: return -1\n        if vb is None: return 1\n        return -1 if va > vb else 1\n    k=COL_ATTEND; va=a[k]; vb=b[k]; \n    if va != vb: \n        return -1 if va < vb else 1\n    return 0\nrows_sorted = sorted(rows_tuple, key=cmp_func(cmp_student), reverse=True)\n```\n\n### Tests / Samples\nName|Descr|Other\n---|---|---\ntests/test_msorted.py|msorted unit tests|- \ntests/performance_tests.py|Tunable performance tests using asyncio | requires pandas\ntests/hand_test.py|Hand testing|-\n',
    'author': 'Timothy C. Quinn',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://pypi.org/project/multisort',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7.9,<4.0.0',
}


setup(**setup_kwargs)
