# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['sqlalchemy_declarative_extensions',
 'sqlalchemy_declarative_extensions.alembic',
 'sqlalchemy_declarative_extensions.dialects',
 'sqlalchemy_declarative_extensions.dialects.mysql',
 'sqlalchemy_declarative_extensions.dialects.postgresql',
 'sqlalchemy_declarative_extensions.dialects.sqlite',
 'sqlalchemy_declarative_extensions.function',
 'sqlalchemy_declarative_extensions.grant',
 'sqlalchemy_declarative_extensions.role',
 'sqlalchemy_declarative_extensions.row',
 'sqlalchemy_declarative_extensions.schema',
 'sqlalchemy_declarative_extensions.trigger',
 'sqlalchemy_declarative_extensions.view']

package_data = \
{'': ['*']}

install_requires = \
['sqlalchemy>=1.3']

extras_require = \
{':python_version < "3.11"': ['typing_extensions>=4.0'],
 'alembic': ['alembic>=1.0'],
 'parse': ['sqlglot']}

setup_kwargs = {
    'name': 'sqlalchemy-declarative-extensions',
    'version': '0.6.0',
    'description': 'Library to declare additional kinds of objects not natively supported by SqlAlchemy/Alembic.',
    'long_description': '# SqlAlchemy Declarative Extensions\n\n[![Actions Status](https://github.com/dancardin/sqlalchemy-declarative-extensions/workflows/test/badge.svg)](https://github.com/dancardin/sqlalchemy-declarative-extensions/actions) [![Coverage Status](https://coveralls.io/repos/github/DanCardin/sqlalchemy-declarative-extensions/badge.svg?branch=main)](https://coveralls.io/github/DanCardin/sqlalchemy-declarative-extensions?branch=main) [![Documentation Status](https://readthedocs.org/projects/sqlalchemy-declarative-extensions/badge/?version=latest)](https://sqlalchemy-declarative-extensions.readthedocs.io/en/latest/?badge=latest)\n\nSee the full documentation [here](https://sqlalchemy-declarative-extensions.readthedocs.io/en/latest/).\n\nAdds extensions to SQLAlchemy (and/or Alembic) which allows declaratively\nstating the existence of additional kinds of objects about your database\nnot natively supported by SqlAlchemy/Alembic.\n\nThis includes:\n\n- Schemas\n- Views\n- Roles\n- Privileges (Grants/Default Grants)\n- Functions\n- Triggers\n- Rows (i.e. data)\n- "audit tables" (i.e. triggers which record data changes to some source table)\n\nThe primary function(s) of this library include:\n\n- Registering onto the SqlAlchemy event system such that `metadata.create_all`\n  creates these objects.\n- (Optionally) Registers into Alembic such that `alembic revision --autogenerate`\n  automatically creates/updates/deletes declared objects.\n\n## Kitchen Sink Example (using all available features)\n\n```python\nfrom sqlalchemy import Column, types, select\nfrom sqlalchemy.orm import as_declarative\nfrom sqlalchemy_declarative_extensions import (\n    declarative_database, Schemas, Roles, Row, View, view,\n)\nfrom sqlalchemy_declarative_extensions.dialects.postgresql import (\n    DefaultGrant, Function, Trigger, Role\n)\nfrom sqlalchemy_declarative_extensions.audit import audit\n\n\n@declarative_database\n@as_declarative\nclass Base:\n    # Note: each object type also has a plural version (i.e. Schemas/Roles/etc) where you can specify\n    # collection-level options like `ignore_unspecified`).\n    #\n    # If you dont set any collection-level options, you can instead use raw list/iterable as the collection.\n    schemas = Schemas().are("example")\n    roles = Roles(ignore_unspecified=True).are(\n        Role("read", login=False),\n        Role(\n            "app",\n            in_roles=[\'read\']\n        ),\n    )\n    grants = [\n        DefaultGrant.on_tables_in_schema("public", \'example\').grant("select", to="read"),\n        DefaultGrant.on_sequences_in_schema("public").grant("usage", to="read"),\n        Grant.new("usage", to="read").on_schemas("example")\n    ]\n    rows = [\n        Row(\'foo\', id=1),\n    ]\n    views = [\n        View("low_foo", "select * from foo where i < 10"),\n    ]\n    functions = [\n        Function(\n            "fancy_function",\n            """\n            BEGIN\n            INSERT INTO foo (id) select NEW.id + 1;\n            RETURN NULL;\n            END\n            """,\n            language="plpgsql",\n            returns="trigger",\n        ),\n    ]\n    triggers = [\n        Trigger.after("insert", on="foo", execute="fancy_function")\n        .named("on_insert_foo")\n        .when("pg_trigger_depth() < 1")\n        .for_each_row(),\n    ]\n\n\n@audit()\nclass Foo(Base):\n    __tablename__ = \'foo\'\n\n    id = Column(types.Integer(), primary_key=True)\n\n\naudit_table = Foo.__audit_table__\n\n\n@view(Base)\nclass HighFoo:\n    __tablename__ = "high_foo"\n    __view__ = select(Foo.__table__).where(Foo.__table__.c.id >= 10)\n```\n\nNote, there is also support for declaring objects directly through the `MetaData` for\nusers not using sqlalchemy\'s declarative API.\n\n## Event Registration\n\nBy default, the above example does not automatically do anything. Depending on the context,\nyou can use one of two registration hooks: `register_sqlalchemy_events` or `register_alembic_events`.\n\n### `register_sqlalchemy_events`\n\nThis registers events in SqlAlchemy\'s event system such that a `metadata.create_all(...)` call\nwill create the declared database objects.\n\n```python\nfrom sqlalchemy_declarative_extensions import register_sqlalchemy_events\n\nmetadata = Base.metadata  # Given the example above.\nregister_sqlalchemy_events(metadata)\n# Which is equivalent to...\nregister_sqlalchemy_events(metadata, schemas=False, roles=False, grants=False, rows=False)\n```\n\nAll object types are opt in, and should be explicitly included in order to get registered.\n\nPractically, this is to avoid issues with testing. In **most** cases the `metadata.create_all` call\nwill be run in tests, a context where it\'s more likely that you dont **need** grants or grants,\nand where parallel test execution could lead to issues with role or schema creation, depending\non your setup.\n\n### `register_alembic_events`\n\nThis registers comparators in Alembic\'s registration system such that an `alembic revision --autogenerate`\ncommand will diff the existing database state against the declared database objects, and issue\nstatements to create/update/delete objects in order to match the declared state.\n\n```python\n# alembic\'s `env.py`\nfrom sqlalchemy_declarative_extensions import register_alembic_events\n\nregister_alembic_events()\n# Which is equivalent to...\nregister_sqlalchemy_events(schemas=True, roles=True, grants=True, rows=True)\n```\n\nAll object types are opt out but can be excluded.\n\nIn contrast to `register_sqlalchemy_events`, it\'s much more likely that you\'re declaring most of these\nobject types in order to have alembic track them\n\n## Database support\n\nIn principle, this library **can** absolutely support any database supported by SqlAlchemy,\nand capable of being introspected enough to support detection of different kinds of objects.\n\nAs you can see below, in reality the existence of implementations are going to be purely driven by actual\nusage. The current maintainer(s) primarily use PostgreSQL and as such individual features for\nother databases will either suffer or lack implementation.\n\n| | Postgres | MySQL | SQLite |\n| Schema | ✓ | | ✓ |\n| View | ✓ | ✓ | ✓ |\n| Role | ✓ | | |\n| Grant | ✓ | | |\n| Default Grant | ✓ | | |\n| Function | ✓ | _ | |\n| Trigger | ✓ | _ | |\n| Row (data) | ✓ | ✓ | ✓ |\n| "Audit Table" | ✓ | | |\n\nThe astrisks above note pending or provisional support. The level of expertise in each dialects\'\nparticular behaviors is not uniform, and deciding on the correct behavior for those dialects\nwill require users to submit issues/fixes!\n\nSupporting a new dialect **can** be as simple as providing the dialect-dispatched implementations\nfor detecting existing objects of the given type. Very much the intent is that once a given object\ntype is supported at all, the comparison infrastructure for that type should make it straightforward\nto support other dialects. At the end of the day, this library is primarily producing SQL statements,\nso in theory any dialect supporting a given object type should be able to be supported.\n\nIn such cases (like Grants/Roles) that different dialects support wildly different\noptions/syntax, there are also patterns for defining dialect-specific objects, to mediate\nany additional differences.\n\n## Alembic-utils\n\n[Alembic Utils](https://github.com/olirice/alembic_utils) is the primary library against which\nthis library can be compared. At time of writing, **most** (but not all) object types supported\nby alembic-utils are supported by this library. One might begin to question when to use which library.\n\nBelow is a list of points on which the two libraries diverge. But note that you **can** certainly\nuse both in tandem! It doesn\'t need to be one or the other, and certainly for any object types\nwhich do not overlap, you might **need** to use both.\n\n- Database Support\n\n  - Alembic Utils seems to explicitly only support PostgreSQL.\n\n  - This library is designed to support any dialect (in theory). Certainly PostgreSQL\n    is **best** supported, but there does exist support for specific kinds of objects\n    to varying levels of support for SQLite and MySQL, so far.\n\n- Architecture\n\n  - Alembic Utils is directly tied to Alembic and does not support SQLAlchemy\'s `MetaData.create_all`.\n    It\'s also the responsibility of the user to discover/register objects in alembic.\n\n  - This library **depends** only on SqlAlchemy, although it also supports alembic. Support for\n    `MetaData.create_all` can be important for creating all object types in tests. It also\n    is designed such that objects are registered on the `MetaData` itself, so there is no need for\n    any specific discovery phase.\n\n- Scope\n\n  - Alembic Utils declares specific, individual objects. I.e. you instantiate one specific `PGGrantTable`\n    or `PGView` instance and Alembic know knows you want that object to be created. It cannot drop\n    objects it is not already aware of.\n\n  - This library declares ths objects the system as a whole expects to exist. Similar to Alembic\'s\n    behavior on tables, it will (by default) detect any **undeclared** objects that should not exist\n    and drop them. That means, you can rely on this object to ensure the state of your migrations\n    matches the state of your database exactly.\n\n- Migration history\n\n  - Alembic Utils imports and references its own objects in your migrations history. This can be\n    unfortunate, in that it deeply ties your migrations history to alembic-utils.\n\n    (In fact, this can be a sticking point, trying to convert **away** from `alembic_utils`, because it\n    will attempt to drop all the (e.g `PGView`) instances previously created when we replaced it with\n    this library.)\n\n  - This library, by contrast, prefers to emit the raw SQL of the operation into your migration.\n    That means you know the exact commands that will execute in your migration, which can be helpful\n    in debugging failure. It also means, if at any point you decide to stop use of the library\n    (or pause a given type of object, due to a bug), you can! This library\'s behaviors are primarily\n    very much `--autogenerate`-time only.\n\n- Abstraction Level\n\n  - Alembic Utils appears to define a very "literal" interface (for example, `PGView` accepts\n    the whole view definition as a raw literal string).\n\n  - This library tries to, as much as possible, provide a more abstracted interface that enables\n    more compatibility with SQLAlchemy (For example `View` accepts SQLAlchemy objects which can\n    be coerced into a `SELECT`). It also tends towards "builder" interfaces which progressively produce\n    a object (Take a look at the `DefaultGrant` above, for an example of where that\'s helpful).\n\nA separate note on conversion/compatibility. Where possible, this library tries to support alembic-utils\nnative objects as stand-ins for the objects defined in this library. For example, `alembic_utils.pg_view.PGView`\ncan be declared instead of a `sqlalchemy_declarative_extensions.View`, and we will internally\ncoerce it into the appropriate type. Hopefully this eases any transitional costs, or\nissues using one or the other library.\n',
    'author': 'Dan Cardin',
    'author_email': 'ddcardin@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/dancardin/sqlalchemy-declarative-extensions',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4',
}


setup(**setup_kwargs)
