"""
Yugang Created at Aug 08, 2016, CHX-NSLS-II

Create a PDF file from XPCS data analysis results, which are generated by CHX data analysis pipeline

How to use: 
python Create_Report.py  full_file_path uid  output_dir (option)

An exmplae to use:
python Create_Report.py  /XF11ID/analysis/2016_2/yuzhang/Results/August/af8f66/ af8f66

python Create_Report.py  /XF11ID/analysis/2016_2/yuzhang/Results/August/af8f66/ af8f66  /XF11ID/analysis/2016_2/yuzhang/Results/August/af8f66/test/

"""


def check_dict_keys(dicts, key):
    if key not in list(dicts.keys()):
        dicts[key] = "unknown"


import h5py

from reportlab.pdfgen import canvas
from reportlab.lib.units import inch, cm, mm
from reportlab.lib.colors import pink, green, brown, white, black, red, blue


from reportlab.lib.styles import getSampleStyleSheet

# from reportlab.platypus import Image, Paragraph, Table

from reportlab.lib.pagesizes import letter, A4
from pyCHX.chx_generic_functions import pload_obj


from PIL import Image
from time import time
from datetime import datetime

import sys, os
import pandas as pds
import numpy as np


def add_one_line_string(c, s, top, left=30, fontsize=11):
    if (fontsize * len(s)) > 1000:
        fontsize = 1000.0 / (len(s))
    c.setFont("Helvetica", fontsize)
    c.drawString(left, top, s)


def add_image_string(
    c,
    imgf,
    data_dir,
    img_left,
    img_top,
    img_height,
    str1_left,
    str1_top,
    str1,
    str2_left,
    str2_top,
    return_=False,
):

    image = data_dir + imgf
    if os.path.exists(image):
        im = Image.open(image)
        ratio = float(im.size[1]) / im.size[0]
        height = img_height
        width = height / ratio
        # if width>400:
        #    width = 350
        #    height = width*ratio
        c.drawImage(image, img_left, img_top, width=width, height=height, mask=None)

        c.setFont("Helvetica", 16)
        c.setFillColor(blue)
        c.drawString(str1_left, str1_top, str1)
        c.setFont("Helvetica", 12)
        c.setFillColor(red)
        c.drawString(str2_left, str2_top, "filename: %s" % imgf)
        if return_:
            return height / ratio

    else:
        c.setFillColor(blue)
        c.drawString(str1_left, str1_top, str1)
        c.setFillColor(red)
        c.drawString(str1_left, str1_top - 40, "-->Not Calculated!")


class create_pdf_report(object):

    """Aug 16, YG@CHX-NSLS-II
     Create a pdf report by giving data_dir, uid, out_dir
     data_dir: the input data directory, including all necessary images
     the images names should be:
          meta_file = 'uid=%s-md'%uid
          avg_img_file = 'uid=%s--img-avg-.png'%uid
          ROI_on_img_file = 'uid=%s--ROI-on-Image-.png'%uid
          qiq_file = 'uid=%s--Circular-Average-.png'%uid
          ROI_on_Iq_file = 'uid=%s--ROI-on-Iq-.png'%uid

          Iq_t_file = 'uid=%s--Iq-t-.png'%uid
          img_sum_t_file = 'uid=%s--img-sum-t.png'%uid
          wat_file= 'uid=%s--Waterfall-.png'%uid
          Mean_inten_t_file= 'uid=%s--Mean-intensity-of-each-ROI-.png'%uid

          g2_file = 'uid=%s--g2-.png'%uid
          g2_fit_file = 'uid=%s--g2--fit-.png'%uid
          q_rate_file = 'uid=--%s--Q-Rate--fit-.png'%uid

          two_time_file = 'uid=%s--Two-time-.png'%uid
          two_g2_file = 'uid=%s--g2--two-g2-.png'%uid

    uid: the unique id
    out_dir: the output directory
    report_type:
        'saxs':  report saxs results
        'gisaxs': report gisaxs results


    Output:
        A PDF file with name as "XPCS Analysis Report for uid=%s"%uid in out_dir folder
    """

    def __init__(
        self,
        data_dir,
        uid,
        out_dir=None,
        filename=None,
        load=True,
        user=None,
        report_type="saxs",
        md=None,
        res_h5_filename=None,
    ):
        self.data_dir = data_dir
        self.uid = uid
        self.md = md
        # print(md)
        if user is None:
            user = "chx"
        self.user = user
        if out_dir is None:
            out_dir = data_dir
        if not os.path.exists(out_dir):
            os.makedirs(out_dir)
        self.out_dir = out_dir

        self.styles = getSampleStyleSheet()
        self.width, self.height = letter

        self.report_type = report_type
        dt = datetime.now()
        CurTime = "%02d/%02d/%s/-%02d/%02d/" % (
            dt.month,
            dt.day,
            dt.year,
            dt.hour,
            dt.minute,
        )
        self.CurTime = CurTime
        if filename is None:
            filename = "XPCS_Analysis_Report_for_uid=%s.pdf" % uid
        filename = out_dir + filename
        c = canvas.Canvas(filename, pagesize=letter)
        self.filename = filename
        self.res_h5_filename = res_h5_filename
        # c.setTitle("XPCS Analysis Report for uid=%s"%uid)
        c.setTitle(filename)
        self.c = c
        if load:
            self.load_metadata()

    def load_metadata(self):
        uid = self.uid
        data_dir = self.data_dir
        # load metadata
        meta_file = "uid=%s_md" % uid
        self.metafile = data_dir + meta_file
        if self.md is None:
            md = pload_obj(data_dir + meta_file)
            self.md = md
        else:
            md = self.md
            # print('Get md from giving md')
            # print(md)
        self.sub_title_num = 0
        uid_g2 = None
        uid_c12 = None
        if "uid_g2" in list(md.keys()):
            uid_g2 = md["uid_g2"]
        if "uid_c12" in list(md.keys()):
            uid_c12 = md["uid_c12"]

        """global definition"""

        if "beg_OneTime" in list(md.keys()):
            beg_OneTime = md["beg_OneTime"]
            end_OneTime = md["end_OneTime"]
        else:
            beg_OneTime = None
            end_OneTime = None

        if "beg_TwoTime" in list(md.keys()):
            beg_TwoTime = md["beg_TwoTime"]
            end_TwoTime = md["end_TwoTime"]
        else:
            beg_TwoTime = None
            end_TwoTime = None

        try:
            beg = md["beg"]
            end = md["end"]
            uid_ = uid + "_fra_%s_%s" % (beg, end)
            if beg_OneTime is None:
                uid_OneTime = uid + "_fra_%s_%s" % (beg, end)
            else:
                uid_OneTime = uid + "_fra_%s_%s" % (beg_OneTime, end_OneTime)
            if beg_TwoTime is None:
                uid_TwoTime = uid + "_fra_%s_%s" % (beg, end)
            else:
                uid_TwoTime = uid + "_fra_%s_%s" % (beg_TwoTime, end_TwoTime)

        except:
            uid_ = uid
            uid_OneTime = uid
        if beg is None:
            uid_ = uid
            uid_OneTime = uid

        self.avg_img_file = "uid=%s_img_avg.png" % uid
        self.ROI_on_img_file = "uid=%s_ROI_on_Image.png" % uid

        self.qiq_file = "uid=%s_q_Iq.png" % uid
        self.qiq_fit_file = "uid=%s_form_factor_fit.png" % uid
        # self.qr_1d_file = 'uid=%s_Qr_ROI.png'%uid
        if self.report_type == "saxs" or self.report_type == "ang_saxs":
            self.ROI_on_Iq_file = "uid=%s_ROI_on_Iq.png" % uid

        elif self.report_type == "gi_saxs":
            self.ROI_on_Iq_file = "uid=%s_Qr_ROI.png" % uid

        self.Iq_t_file = "uid=%s_q_Iqt.png" % uid
        self.img_sum_t_file = "uid=%s_img_sum_t.png" % uid
        self.wat_file = "uid=%s_waterfall.png" % uid
        self.Mean_inten_t_file = "uid=%s_t_ROIs.png" % uid
        self.oavs_file = "uid=%s_OAVS.png" % uid

        if uid_g2 is None:
            uid_g2 = uid_OneTime
        self.g2_file = "uid=%s_g2.png" % uid_g2
        self.g2_fit_file = "uid=%s_g2_fit.png" % uid_g2
        # print(  self.g2_fit_file )
        self.g2_new_page = False
        self.g2_fit_new_page = False
        if self.report_type == "saxs":
            jfn = "uid=%s_g2.png" % uid_g2
            if os.path.exists(data_dir + jfn):
                self.g2_file = jfn
            else:
                jfn = "uid=%s_g2__joint.png" % uid_g2
                if os.path.exists(data_dir + jfn):
                    self.g2_file = jfn
                    self.g2_new_page = True
                # self.g2_new_page = True
            jfn = "uid=%s_g2_fit.png" % uid_g2
            if os.path.exists(data_dir + jfn):
                self.g2_fit_file = jfn
                # self.g2_fit_new_page = True
            else:
                jfn = "uid=%s_g2_fit__joint.png" % uid_g2
                if os.path.exists(data_dir + jfn):
                    self.g2_fit_file = jfn
                    self.g2_fit_new_page = True

        else:
            jfn = "uid=%s_g2__joint.png" % uid_g2
            if os.path.exists(data_dir + jfn):
                self.g2_file = jfn
                self.g2_new_page = True
            jfn = "uid=%s_g2_fit__joint.png" % uid_g2
            if os.path.exists(data_dir + jfn):
                self.g2_fit_file = jfn
                self.g2_fit_new_page = True

        self.q_rate_file = "uid=%s_Q_Rate_fit.png" % uid_g2
        self.q_rate_loglog_file = "uid=%s_Q_Rate_loglog.png" % uid_g2
        self.g2_q_fitpara_file = "uid=%s_g2_q_fitpara_plot.png" % uid_g2

        # print( self.q_rate_file )
        if uid_c12 is None:
            uid_c12 = uid_
        self.q_rate_two_time_fit_file = "uid=%s_two_time_Q_Rate_fit.png" % uid_c12
        # print(  self.q_rate_two_time_fit_file )

        self.two_time_file = "uid=%s_Two_time.png" % uid_c12
        self.two_g2_file = "uid=%s_g2_two_g2.png" % uid_c12

        if self.report_type == "saxs":

            jfn = "uid=%s_g2_two_g2.png" % uid_c12
            self.two_g2_new_page = False
            if os.path.exists(data_dir + jfn):
                # print( 'Here we go')
                self.two_g2_file = jfn
                # self.two_g2_new_page = True
            else:
                jfn = "uid=%s_g2_two_g2__joint.png" % uid_c12
                self.two_g2_new_page = False
                if os.path.exists(data_dir + jfn):
                    # print( 'Here we go')
                    self.two_g2_file = jfn
                    self.two_g2_new_page = True
        else:
            jfn = "uid=%s_g2_two_g2__joint.png" % uid_c12
            self.two_g2_new_page = False
            if os.path.exists(data_dir + jfn):
                # print( 'Here we go')
                self.two_g2_file = jfn
                self.two_g2_new_page = True

        self.four_time_file = "uid=%s_g4.png" % uid_
        jfn = "uid=%s_g4__joint.png" % uid_
        self.g4_new_page = False
        if os.path.exists(data_dir + jfn):
            self.four_time_file = jfn
            self.g4_new_page = True

        self.xsvs_fit_file = "uid=%s_xsvs_fit.png" % uid_
        self.contrast_file = "uid=%s_contrast.png" % uid_
        self.dose_file = "uid=%s_dose_analysis.png" % uid_

        jfn = "uid=%s_dose_analysis__joint.png" % uid_
        self.dose_file_new_page = False
        if os.path.exists(data_dir + jfn):
            self.dose_file = jfn
            self.dose_file_new_page = True

        # print( self.dose_file )
        if False:
            self.flow_g2v = "uid=%s_1a_mqv_g2_v_fit.png" % uid_
            self.flow_g2p = "uid=%s_1a_mqp_g2_p_fit.png" % uid_
            self.flow_g2v_rate_fit = "uid=%s_v_fit_rate_Q_Rate_fit.png" % uid_
            self.flow_g2p_rate_fit = "uid=%s_p_fit_rate_Q_Rate_fit.png" % uid_

        if True:
            self.two_time = "uid=%s_pv_two_time.png" % uid_
            # self.two_time_v = 'uid=%s_pv_two_time.png'%uid_

            # self.flow_g2bv = 'uid=%s_g2b_v_fit.png'%uid_
            # self.flow_g2bp = 'uid=%s_g2b_p_fit.png'%uid_
            self.flow_g2_g2b_p = "uid=%s_g2_two_g2_p.png" % uid_
            self.flow_g2_g2b_v = "uid=%s_g2_two_g2_v.png" % uid_

            self.flow_g2bv_rate_fit = "uid=%s_vertb_Q_Rate_fit.png" % uid_
            self.flow_g2bp_rate_fit = "uid=%s_parab_Q_Rate_fit.png" % uid_

            self.flow_g2v = "uid=%s_g2_v_fit.png" % uid_
            self.flow_g2p = "uid=%s_g2_p_fit.png" % uid_
            self.flow_g2v_rate_fit = "uid=%s_vert_Q_Rate_fit.png" % uid_
            self.flow_g2p_rate_fit = "uid=%s_para_Q_Rate_fit.png" % uid_

        # self.report_header(page=1, top=730, new_page=False)
        # self.report_meta(new_page=False)

        self.q2Iq_file = "uid=%s_q2_iq.png" % uid
        self.iq_invariant_file = "uid=%s_iq_invariant.png" % uid

    def report_invariant(self, top=300, new_page=False):
        """create the invariant analysis report
        two images:
           ROI on average intensity image
           ROI on circular average
        """
        uid = self.uid
        c = self.c
        # add sub-title, static images
        c.setFillColor(black)
        c.setFont("Helvetica", 20)
        ds = 230
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. I(q) Invariant Analysis" % self.sub_title_num
        )  # add title
        # add q2Iq
        c.setFont("Helvetica", 14)
        imgf = self.q2Iq_file
        # print( imgf )
        label = "q^2*I(q)"
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left=60,
            img_top=top - ds * 1.15,
            img_height=180,
            str1_left=110,
            str1_top=top - 35,
            str1=label,
            str2_left=60,
            str2_top=top - 320,
        )

        # add iq_invariant
        imgf = self.iq_invariant_file
        img_height = 180
        img_left, img_top = 320, top - ds * 1.15
        str1_left, str1_top, str1 = 420, top - 35, "I(q) Invariant"
        str2_left, str2_top = 350, top - 320

        # print ( imgf )

        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        if new_page:
            c.showPage()
            c.save()

    def report_header(self, page=1, new_page=False):
        """create headers, including title/page number"""
        c = self.c
        CurTime = self.CurTime
        uid = self.uid
        user = self.user
        c.setFillColor(black)
        c.setFont("Helvetica", 14)
        # add page number
        c.drawString(250, 10, "Page--%s--" % (page))
        # add time stamp

        # c.drawString(350, 10, "Created at %s@CHX-by-%s"%( CurTime,user ) )
        s_ = "Created at %s@CHX-By-%s" % (CurTime, user)
        add_one_line_string(c, s_, 10, left=350, fontsize=11)

        # add title
        # c.setFont("Helvetica", 22)
        title = "XPCS Analysis Report for uid=%s" % uid
        c.setFont("Helvetica", 1000 / (len(title)))
        # c.drawString(180,760, "XPCS Report of uid=%s"%uid )  #add title
        c.drawString(50, 760, "XPCS Analysis Report for uid=%s" % uid)  # add title
        # add a line under title
        c.setStrokeColor(red)
        c.setLineWidth(width=1.5)
        c.line(50, 750, 550, 750)
        if new_page:
            c.showPage()
            c.save()

    def report_meta(self, top=740, new_page=False):
        """create the meta data report,
        the meta data include:
            uid
            Sample:
            Measurement
            Wavelength
            Detector-Sample Distance
            Beam Center
            Mask file
            Data dir
            Pipeline notebook
        """

        c = self.c
        # load metadata
        md = self.md
        try:
            uid = md["uid"]
        except:
            uid = self.uid
        # add sub-title, metadata
        c.setFont("Helvetica", 20)
        ds = 15
        self.sub_title_num += 1
        c.drawString(10, top, "%s. Metadata" % self.sub_title_num)  # add title
        top = top - 5
        fontsize = 11
        c.setFont("Helvetica", fontsize)

        nec_keys = [
            "sample",
            "start_time",
            "stop_time",
            "Measurement",
            "exposure time",
            "incident_wavelength",
            "cam_acquire_t",
            "frame_time",
            "detector_distance",
            "feedback_x",
            "feedback_y",
            "shutter mode",
            "beam_center_x",
            "beam_center_y",
            "beam_refl_center_x",
            "beam_refl_center_y",
            "mask_file",
            "bad_frame_list",
            "transmission",
            "roi_mask_file",
        ]
        for key in nec_keys:
            check_dict_keys(md, key)

        try:  # try exp time from detector
            exposuretime = md["count_time"]  # exposure time in sec
        except:
            exposuretime = md["cam_acquire_time"]  # exposure time in sec

        try:  # try acq time from detector
            acquisition_period = md["frame_time"]
        except:
            try:
                acquisition_period = md["acquire period"]
            except:
                uid = md["uid"]
                acquisition_period = float(db[uid]["start"]["acquire period"])

        s = []
        s.append("UID: %s" % uid)  ###line 1, for uid
        s.append("Sample: %s" % md["sample"])  ####line 2 sample
        s.append(
            "Data Acquisition From: %s To: %s" % (md["start_time"], md["stop_time"])
        )  ####line 3 Data Acquisition time
        s.append("Measurement: %s" % md["Measurement"])  ####line 4 'Measurement

        # print(  md['incident_wavelength'],  int(md['number of images']),              md['detector_distance'], md['feedback_x'], md['feedback_y'], md['shutter mode']  )
        # print(acquisition_period)
        s.append(
            "Wavelength: %s A | Num of Image: %d | Exposure time: %s ms | Acquire period: %s ms"
            % (
                md["incident_wavelength"],
                int(md["number of images"]),
                round(float(exposuretime) * 1000, 4),
                round(float(acquisition_period) * 1000, 4),
            )
        )  ####line 5 'lamda...

        s.append(
            "Detector-Sample Distance: %s m| FeedBack Mode: x -> %s & y -> %s| Shutter Mode: %s"
            % (
                md["detector_distance"],
                md["feedback_x"],
                md["feedback_y"],
                md["shutter mode"],
            )
        )  ####line 6 'Detector-Sample Distance..
        if self.report_type == "saxs":
            s7 = "Beam Center: [%s, %s] (pixel)" % (
                md["beam_center_x"],
                md["beam_center_y"],
            )
        elif self.report_type == "gi_saxs":
            s7 = (
                "Incident Center: [%s, %s] (pixel)"
                % (md["beam_center_x"], md["beam_center_y"])
                + "   ||   "
                + "Reflect Center: [%s, %s] (pixel)"
                % (md["beam_refl_center_x"], md["beam_refl_center_y"])
            )
        elif self.report_type == "ang_saxs" or self.report_type == "gi_waxs":
            s7 = "Beam Center: [%s, %s] (pixel)" % (
                md["beam_center_x"],
                md["beam_center_y"],
            )
        else:
            s7 = ""

        s7 += " || " + "BadLen: %s" % len(md["bad_frame_list"])
        s7 += " || " + "Transmission: %s" % md["transmission"]
        s.append(s7)  ####line 7 'Beam center...
        m = (
            "Mask file: %s" % md["mask_file"]
            + " || "
            + "ROI mask file: %s" % md["roi_mask_file"]
        )
        # s.append(   'Mask file: %s'%md['mask_file'] )  ####line 8 mask filename
        # s.append(    )  ####line 8 mask filename
        s.append(m)

        if self.res_h5_filename is not None:
            self.data_dir_ = self.data_dir + self.res_h5_filename
        else:
            self.data_dir_ = self.data_dir
        s.append("Analysis Results Dir: %s" % self.data_dir_)  ####line 9 results folder

        s.append(
            "Metadata Dir: %s.csv-&.pkl" % self.metafile
        )  ####line 10 metadata folder
        try:
            s.append(
                "Pipeline notebook: %s" % md["NOTEBOOK_FULL_PATH"]
            )  ####line 11 notebook folder
        except:
            pass
        # print( 'here' )
        line = 1
        for s_ in s:
            add_one_line_string(c, s_, top - ds * line, left=30, fontsize=fontsize)
            line += 1

        if new_page:
            c.showPage()
            c.save()

    def report_static(self, top=560, new_page=False, iq_fit=False):
        """create the static analysis report
        two images:
            average intensity image
            circular average

        """
        # add sub-title, static images

        c = self.c
        c.setFont("Helvetica", 20)
        uid = self.uid

        ds = 220
        self.sub_title_num += 1
        c.drawString(10, top, "%s. Static Analysis" % self.sub_title_num)  # add title

        # add average image
        c.setFont("Helvetica", 14)

        imgf = self.avg_img_file

        if self.report_type == "saxs":
            ipos = 60
            dshift = 0
        elif self.report_type == "gi_saxs":
            ipos = 200
            dshift = 140
        elif self.report_type == "ang_saxs":
            ipos = 200
            dshift = 140
        else:
            ipos = 200
            dshift = 140

        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left=ipos,
            img_top=top - ds,
            img_height=180,
            str1_left=90 + dshift,
            str1_top=top - 35,
            str1="Average Intensity Image",
            str2_left=80 + dshift,
            str2_top=top - 230,
        )

        # add q_Iq
        if self.report_type == "saxs":
            imgf = self.qiq_file
            # print(imgf)
            if iq_fit:
                imgf = self.qiq_fit_file
            label = "Circular Average"
            lab_pos = 390
            fn_pos = 320
            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left=320,
                img_top=top - ds,
                img_height=180,
                str1_left=lab_pos,
                str1_top=top - 35,
                str1=label,
                str2_left=fn_pos,
                str2_top=top - 230,
            )
        else:
            if False:
                imgf = self.ROI_on_Iq_file  # self.qr_1d_file
                label = "Qr-1D"
                lab_pos = 420
                fn_pos = 350

                add_image_string(
                    c,
                    imgf,
                    self.data_dir,
                    img_left=320,
                    img_top=top - ds,
                    img_height=180,
                    str1_left=lab_pos,
                    str1_top=top - 35,
                    str1=label,
                    str2_left=fn_pos,
                    str2_top=top - 230,
                )
        if new_page:
            c.showPage()
            c.save()

    def report_ROI(self, top=300, new_page=False):
        """create the static analysis report
        two images:
           ROI on average intensity image
           ROI on circular average
        """
        uid = self.uid
        c = self.c
        # add sub-title, static images
        c.setFillColor(black)
        c.setFont("Helvetica", 20)
        ds = 230
        self.sub_title_num += 1
        c.drawString(10, top, "%s. Define of ROI" % self.sub_title_num)  # add title
        # add ROI on image
        c.setFont("Helvetica", 14)
        imgf = self.ROI_on_img_file
        label = "ROI on Image"
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left=60,
            img_top=top - ds * 1.15,
            img_height=240,
            str1_left=110,
            str1_top=top - 35,
            str1=label,
            str2_left=60,
            str2_top=top - 260,
        )

        # add q_Iq
        if (
            self.report_type == "saxs"
            or self.report_type == "gi_saxs"
            or self.report_type == "ang_saxs"
        ):
            imgf = self.ROI_on_Iq_file
            img_height = 180
            img_left, img_top = 320, top - ds
            str1_left, str1_top, str1 = 420, top - 35, "ROI on Iq"
            str2_left, str2_top = 350, top - 260

            # print ( imgf )

            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )

        if new_page:
            c.showPage()
            c.save()

    def report_time_analysis(self, top=720, new_page=False):
        """create the time dependent analysis report
        four images:
            each image total intensity as a function of time
            iq~t
            waterfall
            mean intensity of each ROI as a function of time
        """
        c = self.c
        uid = self.uid
        # add sub-title, Time-dependent plot
        c.setFont("Helvetica", 20)
        top1 = top
        ds = 20
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. Time Dependent Plot" % self.sub_title_num
        )  # add title
        c.setFont("Helvetica", 14)

        top = top1 - 160

        # add img_sum_t
        if self.report_type == "saxs":
            ipos = 80
        elif self.report_type == "gi_saxs":
            ipos = 200
        elif self.report_type == "ang_saxs":
            ipos = 200
        else:
            ipos = 200

        imgf = self.img_sum_t_file
        img_height = 140
        img_left, img_top = ipos, top
        str1_left, str1_top, str1 = ipos + 60, top1 - 20, "img sum ~ t"
        str2_left, str2_top = ipos, top - 5

        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        # plot iq~t
        if self.report_type == "saxs":
            imgf = self.Iq_t_file
            image = self.data_dir + imgf

            img_height = 140
            img_left, img_top = 350, top
            str1_left, str1_top, str1 = 420, top1 - 20, "iq ~ t"
            str2_left, str2_top = 360, top - 5

            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )
        elif self.report_type == "gi_saxs":
            pass

        top = top1 - 340
        # add waterfall plot
        imgf = self.wat_file

        img_height = 160
        img_left, img_top = 80, top
        str1_left, str1_top, str1 = 140, top + img_height, "waterfall plot"
        str2_left, str2_top = 80, top - 5

        if self.report_type != "ang_saxs":
            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )
        else:
            pass

        # add mean-intensity of each roi
        imgf = self.Mean_inten_t_file

        img_height = 160
        img_left, img_top = 360, top
        str1_left, str1_top, str1 = 330, top + img_height, "Mean-intensity-of-each-ROI"
        str2_left, str2_top = 310, top - 5
        if self.report_type != "ang_saxs":
            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )
        else:
            pass

        if new_page:
            c.showPage()
            c.save()

    def report_oavs(self, top=350, oavs_file=None, new_page=False):
        """create the oavs images report"""

        c = self.c
        uid = self.uid
        # add sub-title, One Time Correlation Function
        c.setFillColor(black)
        c.setFont("Helvetica", 20)
        ds = 20
        self.sub_title_num += 1
        c.drawString(10, top, "%s. OAVS Images" % self.sub_title_num)  # add title
        c.setFont("Helvetica", 14)
        # add g2 plot
        if oavs_file is None:
            imgf = self.oavs_file
        else:
            imgf = oavs_file
        # print(self.data_dir + imgf)

        if os.path.exists(self.data_dir + imgf):
            im = Image.open(self.data_dir + imgf)
            ratio = float(im.size[1]) / im.size[0]
            img_width = 600
            img_height = img_width * ratio  # img_height
            # width = height/ratio

        if not new_page:
            # img_height= 550
            top = top - 600
            str2_left, str2_top = 80, top - 400
            img_left, img_top = 1, top

        if new_page:
            # img_height= 150
            top = top - img_height - 50
            str2_left, str2_top = 80, top - 50
            img_left, img_top = 10, top

        str1_left, str1_top, str1 = 150, top + img_height, "OAVS images"
        img_width = add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
            return_=True,
        )
        # print( imgf,self.data_dir )
        print(img_width, img_height)

    def report_one_time(
        self, top=350, g2_fit_file=None, q_rate_file=None, new_page=False
    ):
        """create the one time correlation function report
        Two images:
            One Time Correlation Function with fit
            q-rate fit
        """

        c = self.c
        uid = self.uid
        # add sub-title, One Time Correlation Function
        c.setFillColor(black)
        c.setFont("Helvetica", 20)
        ds = 20
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. One Time Correlation Function" % self.sub_title_num
        )  # add title
        c.setFont("Helvetica", 14)
        # add g2 plot
        if g2_fit_file is None:
            imgf = self.g2_fit_file
        else:
            imgf = g2_fit_file

        if self.report_type != "ang_saxs":
            img_height = 300
            top = top - 320
            str2_left, str2_top = 80, top - 0

        else:
            img_height = 550
            top = top - 600
            str2_left, str2_top = 80, top - 400
        # add one_time caculation
        img_left, img_top = 1, top
        if self.g2_fit_new_page or self.g2_new_page:

            img_height = 550
            top = top - 250
            str2_left, str2_top = 80, top - 0
            img_left, img_top = 60, top

        str1_left, str1_top, str1 = 150, top + img_height, "g2 fit plot"
        img_width = add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
            return_=True,
        )
        # print( imgf,self.data_dir )
        # add g2 plot fit
        # print(self.q_rate_file  )
        if os.path.isfile(self.data_dir + self.q_rate_file):
            # print('here')
            # print(self.q_rate_file  )
            top = top + 70  #
            if q_rate_file is None:
                imgf = self.q_rate_file
            else:
                imgf = q_rate_file
            if self.report_type != "ang_saxs":
                # print(img_width)
                if img_width > 400:
                    img_height = 90
                else:
                    img_height = 180
                img_left, img_top = img_width - 10, top  # 350, top
                str2_left, str2_top = img_width + 50, top - 5  # 380, top - 5
                str1_left, str1_top, str1 = 450, top + 230, "q-rate fit  plot"
            else:
                img_height = 300
                img_left, img_top = 350, top - 150
                str2_left, str2_top = 380, top - 5
                str1_left, str1_top, str1 = 450, top + 180, "q-rate fit  plot"
            if self.g2_fit_new_page or self.g2_new_page:
                top = top - 200
                img_height = 180
                img_left, img_top = 350, top
                str2_left, str2_top = 380, top - 5
                str1_left, str1_top, str1 = 450, top + 230, "q-rate fit  plot"
            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )

        else:
            top = top + 320  #
            if q_rate_file is None:
                imgf = self.q_rate_loglog_file
            else:
                imgf = q_rate_file
            # print(imgf)
            if self.report_type != "ang_saxs":
                # print(img_width)
                if img_width > 400:
                    img_height = 90 / 2
                else:
                    img_height = 180 / 2
                img_left, img_top = img_width - 10, top  # 350, top
                str2_left, str2_top = img_width + 50, top - 5  # 380, top - 5
                str1_left, str1_top, str1 = 450, top + 230, "q-rate loglog  plot"
            else:
                img_height = 300 / 2
                img_left, img_top = 350, top - 150
                str2_left, str2_top = 380, top - 5
                str1_left, str1_top, str1 = 450, top + 180, "q-rate loglog  plot"
            if self.g2_fit_new_page or self.g2_new_page:
                top = top - 200 + 50
                img_height = 180 / 1.5
                img_left, img_top = 350, top
                str2_left, str2_top = 380, top - 5
                str1_left, str1_top, str1 = 450, top + 120, "q-rate loglog  plot"

                # print('here')

            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )

            top = top - 100  #
            if q_rate_file is None:
                imgf = self.g2_q_fitpara_file
            else:
                imgf = q_rate_file
            if self.report_type != "ang_saxs":
                # print(img_width)
                if img_width > 400:
                    img_height = 90
                else:
                    img_height = 180
                img_left, img_top = img_width - 10, top  # 350, top
                str2_left, str2_top = img_width + 50, top - 5  # 380, top - 5
                str1_left, str1_top, str1 = 450, top + 230, "g2 fit para"
            else:
                img_height = 300
                img_left, img_top = 350, top - 150
                str2_left, str2_top = 380, top - 5
                str1_left, str1_top, str1 = 450, top + 180, "g2 fit para"
            if self.g2_fit_new_page or self.g2_new_page:
                top = top - 200
                img_height = 180 * 1.5
                img_left, img_top = 350, top
                str2_left, str2_top = 380, top - 5
                str1_left, str1_top, str1 = 450, top + 280, "g2 fit para"
            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )

        if new_page:
            c.showPage()
            c.save()

    def report_mulit_one_time(self, top=720, new_page=False):
        """create the mulit one time correlation function report
        Two images:
            One Time Correlation Function with fit
            q-rate fit
        """
        c = self.c
        uid = self.uid
        # add sub-title, One Time Correlation Function
        c.setFillColor(black)
        c.setFont("Helvetica", 20)
        ds = 20
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. One Time Correlation Function" % self.sub_title_num
        )  # add title
        c.setFont("Helvetica", 14)
        # add g2 plot
        top = top - 320

        imgf = self.g2_fit_file
        image = self.data_dir + imgf
        if not os.path.exists(image):
            image = self.data_dir + self.g2_file
        im = Image.open(image)
        ratio = float(im.size[1]) / im.size[0]
        height = 300
        c.drawImage(image, 1, top, width=height / ratio, height=height, mask="auto")
        # c.drawImage( image, 1, top,  width= height/ratio,height=height, mask= None )
        c.setFont("Helvetica", 16)
        c.setFillColor(blue)
        c.drawString(150, top + height, "g2 fit plot")

        c.setFont("Helvetica", 12)
        c.setFillColor(red)
        c.drawString(80, top - 0, "filename: %s" % imgf)

        # add g2 plot fit
        top = top + 70  #
        imgf = self.q_rate_file
        image = self.data_dir + imgf
        if os.path.exists(image):
            im = Image.open(image)
            ratio = float(im.size[1]) / im.size[0]
            height = 180
            c.drawImage(
                image, 350, top, width=height / ratio, height=height, mask="auto"
            )

            c.setFont("Helvetica", 16)
            c.setFillColor(blue)
            c.drawString(450, top + 230, "q-rate fit  plot")
            c.setFont("Helvetica", 12)
            c.setFillColor(red)
            c.drawString(380, top - 5, "filename: %s" % imgf)

        if new_page:
            c.showPage()
            c.save()

    def report_two_time(self, top=720, new_page=False):
        """create the one time correlation function report
        Two images:
            Two Time Correlation Function
            two one-time correlatoin function from multi-one-time and from diagonal two-time
        """
        c = self.c
        uid = self.uid
        # add sub-title, Time-dependent plot
        c.setFont("Helvetica", 20)

        ds = 20
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. Two Time Correlation Function" % self.sub_title_num
        )  # add title
        c.setFont("Helvetica", 14)

        top1 = top
        top = top1 - 330
        # add q_Iq_t
        imgf = self.two_time_file

        img_height = 300
        img_left, img_top = 80, top
        str1_left, str1_top, str1 = 180, top + 300, "two time correlation function"
        str2_left, str2_top = 180, top - 10
        img_width = add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
            return_=True,
        )

        top = top - 340
        # add q_Iq_t
        imgf = self.two_g2_file

        if True:  # not self.two_g2_new_page:

            img_height = 300
            img_left, img_top = 100 - 70, top
            str1_left, str1_top, str1 = 210 - 70, top + 310, "compared g2"
            str2_left, str2_top = 180 - 70, top - 10

            if self.two_g2_new_page:
                img_left, img_top = 100, top
            print(imgf)
            img_width = add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
                return_=True,
            )
            # print(imgf)
            top = top + 50
            imgf = self.q_rate_two_time_fit_file
            # print(imgf, img_width, top)
            if img_width < 400:
                img_height = 140
                img_left, img_top = 350, top + 30
                str2_left, str2_top = 380 - 80, top - 5
                str1_left, str1_top, str1 = (
                    450 - 80,
                    top + 230,
                    "q-rate fit from two-time",
                )

            else:
                img_height = 90
                img_left, img_top = img_width - 10, top  # 350, top
                str2_left, str2_top = img_width + 50, top - 5  # 380, top - 5
                str1_left, str1_top, str1 = 450, top + 230, "q-rate fit  plot"

            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )

        if new_page:
            c.showPage()
            c.save()

    def report_four_time(self, top=720, new_page=False):
        """create the one time correlation function report
        Two images:
            Two Time Correlation Function
            two one-time correlatoin function from multi-one-time and from diagonal two-time
        """

        c = self.c
        uid = self.uid
        # add sub-title, Time-dependent plot
        c.setFont("Helvetica", 20)

        ds = 20
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. Four Time Correlation Function" % self.sub_title_num
        )  # add title
        c.setFont("Helvetica", 14)

        top1 = top
        top = top1 - 330
        # add q_Iq_t
        imgf = self.four_time_file

        if not self.g4_new_page:
            img_height = 300
            img_left, img_top = 80, top
            str1_left, str1_top, str1 = 180, top + 300, "four time correlation function"
            str2_left, str2_top = 180, top - 10
        else:
            img_height = 600
            top -= 300
            img_left, img_top = 80, top
            str1_left, str1_top, str1 = (
                180,
                top + 300 - 250,
                "four time correlation function",
            )
            str2_left, str2_top = 180, top - 10

        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        if new_page:
            c.showPage()
            c.save()

    def report_dose(self, top=720, new_page=False):

        c = self.c
        uid = self.uid
        # add sub-title, Time-dependent plot
        c.setFont("Helvetica", 20)
        ds = 20
        self.sub_title_num += 1
        c.drawString(10, top, "%s. Dose Analysis" % self.sub_title_num)  # add title
        c.setFont("Helvetica", 14)

        top1 = top
        top = top1 - 530
        # add q_Iq_t
        imgf = self.dose_file

        img_height = 500
        img_left, img_top = 80, top
        str1_left, str1_top, str1 = 180, top + 500, "dose analysis"
        str2_left, str2_top = 180, top - 10

        # print( self.data_dir + self.dose_file)
        if os.path.exists(self.data_dir + imgf):
            # print( self.dose_file)
            im = Image.open(self.data_dir + imgf)
            ratio = float(im.size[1]) / im.size[0]
            width = img_height / ratio
            # print(width)
            if width > 450:
                img_height = 450 * ratio

        if self.dose_file_new_page:
            # img_left,img_top = 180, top
            img_left, img_top = 100, top

        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        if new_page:
            c.showPage()
            c.save()

    def report_flow_pv_g2(self, top=720, new_page=False):
        """create the one time correlation function report
        Two images:
            Two Time Correlation Function
            two one-time correlatoin function from multi-one-time and from diagonal two-time
        """
        c = self.c
        uid = self.uid
        # add sub-title, Time-dependent plot
        c.setFont("Helvetica", 20)

        ds = 20
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. Flow One Time Analysis" % self.sub_title_num
        )  # add title
        c.setFont("Helvetica", 14)

        top1 = top
        top = top1 - 330
        # add xsvs fit

        imgf = self.flow_g2v
        image = self.data_dir + imgf

        img_height = 300
        img_left, img_top = 80, top
        str1_left, str1_top, str1 = 210, top + 300, "XPCS Vertical Flow"
        str2_left, str2_top = 180, top - 10
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        imgf = self.flow_g2v_rate_fit
        img_height = 200
        img_left, img_top = 350, top + 50
        str1_left, str1_top, str1 = 210, top + 300, ""
        str2_left, str2_top = 350, top - 10 + 50
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        top = top - 340
        # add contrast fit
        imgf = self.flow_g2p
        img_height = 300
        img_left, img_top = 80, top
        str1_left, str1_top, str1 = 210, top + 300, "XPCS Parallel Flow"
        str2_left, str2_top = 180, top - 10
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        imgf = self.flow_g2p_rate_fit
        img_height = 200
        img_left, img_top = 350, top + 50
        str1_left, str1_top, str1 = 210, top + 300, ""
        str2_left, str2_top = 350, top - 10 + 50
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        if new_page:
            c.showPage()
            c.save()

    def report_flow_pv_two_time(self, top=720, new_page=False):
        """create the two time correlation function report
        Two images:
            Two Time Correlation Function
            two one-time correlatoin function from multi-one-time and from diagonal two-time
        """
        c = self.c
        uid = self.uid
        # add sub-title, Time-dependent plot
        c.setFont("Helvetica", 20)

        ds = 20
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. Flow One &Two Time Comparison" % self.sub_title_num
        )  # add title
        c.setFont("Helvetica", 14)

        top1 = top
        top = top1 - 330
        # add xsvs fit

        if False:
            imgf = self.two_time
            image = self.data_dir + imgf

            img_height = 300
            img_left, img_top = 80, top
            str1_left, str1_top, str1 = 210, top + 300, "Two_time"
            str2_left, str2_top = 180, top - 10
            add_image_string(
                c,
                imgf,
                self.data_dir,
                img_left,
                img_top,
                img_height,
                str1_left,
                str1_top,
                str1,
                str2_left,
                str2_top,
            )

        imgf = self.flow_g2_g2b_p
        img_height = 300
        img_left, img_top = 80, top
        str1_left, str1_top, str1 = 210, top + 300, "XPCS Vertical Flow by two-time"
        str2_left, str2_top = 180, top - 10
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        imgf = self.flow_g2bp_rate_fit
        img_height = 200
        img_left, img_top = 350, top + 50
        str1_left, str1_top, str1 = 210, top + 300, ""
        str2_left, str2_top = 350, top - 10 + 50
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        top = top - 340
        # add contrast fit
        imgf = self.flow_g2_g2b_v

        img_height = 300
        img_left, img_top = 80, top
        str1_left, str1_top, str1 = 210, top + 300, "XPCS Parallel Flow by two-time"
        str2_left, str2_top = 180, top - 10
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        imgf = self.flow_g2bv_rate_fit
        img_height = 200
        img_left, img_top = 350, top + 50
        str1_left, str1_top, str1 = 210, top + 300, ""
        str2_left, str2_top = 350, top - 10 + 50
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left,
            img_top,
            img_height,
            str1_left,
            str1_top,
            str1,
            str2_left,
            str2_top,
        )

        if new_page:
            c.showPage()
            c.save()

    def report_xsvs(self, top=720, new_page=False):
        """create the one time correlation function report
        Two images:
            Two Time Correlation Function
            two one-time correlatoin function from multi-one-time and from diagonal two-time
        """
        c = self.c
        uid = self.uid
        # add sub-title, Time-dependent plot
        c.setFont("Helvetica", 20)

        ds = 20
        self.sub_title_num += 1
        c.drawString(
            10, top, "%s. Visibility Analysis" % self.sub_title_num
        )  # add title
        c.setFont("Helvetica", 14)
        top = top - 330
        # add xsvs fit
        imgf = self.xsvs_fit_file
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left=100,
            img_top=top,
            img_height=300,
            str1_left=210,
            str1_top=top + 300,
            str1="XSVS_Fit_by_Negtive_Binomal Function",
            str2_left=180,
            str2_top=top - 10,
        )

        # add contrast fit
        top = top - 340
        imgf = self.contrast_file
        add_image_string(
            c,
            imgf,
            self.data_dir,
            img_left=100,
            img_top=top,
            img_height=300,
            str1_left=210,
            str1_top=top + 310,
            str1="contrast get from xsvs and xpcs",
            str2_left=180,
            str2_top=top - 10,
        )

        if False:
            top1 = top
            top = top1 - 330
            # add xsvs fit
            imgf = self.xsvs_fit_file
            image = self.data_dir + imgf
            im = Image.open(image)
            ratio = float(im.size[1]) / im.size[0]
            height = 300
            c.drawImage(image, 100, top, width=height / ratio, height=height, mask=None)
            c.setFont("Helvetica", 16)
            c.setFillColor(blue)
            c.drawString(210, top + 300, "XSVS_Fit_by_Negtive_Binomal Function")
            c.setFont("Helvetica", 12)
            c.setFillColor(red)
            c.drawString(180, top - 10, "filename: %s" % imgf)
            top = top - 340
            # add contrast fit
            imgf = self.contrast_file
            image = self.data_dir + imgf
            im = Image.open(image)
            ratio = float(im.size[1]) / im.size[0]
            height = 300
            c.drawImage(image, 100, top, width=height / ratio, height=height, mask=None)

            c.setFont("Helvetica", 16)
            c.setFillColor(blue)
            c.drawString(210, top + 310, "contrast get from xsvs and xpcs")
            c.setFont("Helvetica", 12)
            c.setFillColor(red)
            c.drawString(180, top - 10, "filename: %s" % imgf)

        if new_page:
            c.showPage()
            c.save()

    def new_page(self):
        c = self.c
        c.showPage()

    def save_page(self):
        c = self.c
        c.save()

    def done(self):
        out_dir = self.out_dir
        uid = self.uid

        print()
        print("*" * 40)
        print("The pdf report is created with filename as: %s" % (self.filename))
        print("*" * 40)


def create_multi_pdf_reports_for_uids(
    uids, g2, data_dir, report_type="saxs", append_name=""
):
    """Aug 16, YG@CHX-NSLS-II
    Create multi pdf reports for each uid in uids
    uids: a list of uids to be reported
    g2: a dictionary, {run_num: sub_num: g2_of_each_uid}
    data_dir:
    Save pdf report in data dir
    """
    for key in list(g2.keys()):
        i = 1
        for sub_key in list(g2[key].keys()):
            uid_i = uids[key][sub_key]
            data_dir_ = os.path.join(data_dir, "%s/" % uid_i)
            if append_name != "":
                uid_name = uid_i + append_name
            else:
                uid_name = uid_i
            c = create_pdf_report(
                data_dir_,
                uid_i,
                data_dir,
                report_type=report_type,
                filename="XPCS_Analysis_Report_for_uid=%s.pdf" % uid_name,
            )
            # Page one: Meta-data/Iq-Q/ROI
            c.report_header(page=1)
            c.report_meta(top=730)
            # c.report_one_time( top= 500 )
            # c.new_page()
            if report_type == "flow":
                c.report_flow_pv_g2(top=720)
            c.save_page()
            c.done()


def create_one_pdf_reports_for_uids(
    uids, g2, data_dir, filename="all_in_one", report_type="saxs"
):
    """Aug 16, YG@CHX-NSLS-II
    Create one pdf reports for each uid in uids
    uids: a list of uids to be reported
    g2: a dictionary, {run_num: sub_num: g2_of_each_uid}
    data_dir:
    Save pdf report in data dir
    """
    c = create_pdf_report(
        data_dir, uid=filename, out_dir=data_dir, load=False, report_type=report_type
    )
    page = 1

    for key in list(g2.keys()):
        i = 1
        for sub_key in list(g2[key].keys()):
            uid_i = uids[key][sub_key]
            data_dir_ = os.path.join(data_dir, "%s/" % uid_i)

            c.uid = uid_i
            c.data_dir = data_dir_
            c.load_metadata()

            # Page one: Meta-data/Iq-Q/ROI
            c.report_header(page=page)
            c.report_meta(top=730)
            c.report_one_time(top=500)
            c.new_page()
            page += 1
    c.uid = filename
    c.save_page()
    c.done()


def save_res_h5(full_uid, data_dir, save_two_time=False):
    """
    YG. Nov 10, 2016
    save the results to a h5 file
    will save meta data/avg_img/mask/roi (ring_mask or box_mask)/
    will aslo save multi-tau calculated one-time correlation function g2/taus
    will also save two-time derived one-time correlation function /g2b/taus2
    if save_two_time if True, will save two-time correaltion function
    """
    with h5py.File(data_dir + "%s.h5" % full_uid, "w") as hf:
        # write meta data
        meta_data = hf.create_dataset("meta_data", (1,), dtype="i")
        for key in md.keys():
            try:
                meta_data.attrs[key] = md[key]
            except:
                pass

        shapes = md["avg_img"].shape
        avg_h5 = hf.create_dataset("avg_img", data=md["avg_img"])
        mask_h5 = hf.create_dataset("mask", data=md["mask"])
        roi_h5 = hf.create_dataset("roi", data=md["ring_mask"])

        g2_h5 = hf.create_dataset("g2", data=g2)
        taus_h5 = hf.create_dataset("taus", data=taus)

        if save_two_time:
            g12b_h5 = hf.create_dataset("g12b", data=g12b)
        g2b_h5 = hf.create_dataset("g2b", data=g2b)
        taus2_h5 = hf.create_dataset("taus2", data=taus2)


def printname(name):
    print(name)


# f.visit(printname)
def load_res_h5(full_uid, data_dir):
    """YG. Nov 10, 2016
    load results from a h5 file
    will load meta data/avg_img/mask/roi (ring_mask or box_mask)/
    will aslo load multi-tau calculated one-time correlation function g2/taus
    will also load two-time derived one-time correlation function /g2b/taus2
    if save_two_time if True, will load two-time correaltion function

    """
    with h5py.File(data_dir + "%s.h5" % full_uid, "r") as hf:
        meta_data_h5 = hf.get("meta_data")
        meta_data = {}
        for att in meta_data_h5.attrs:
            meta_data[att] = meta_data_h5.attrs[att]
        avg_h5 = np.array(hf.get("avg_img"))
        mask_h5 = np.array(hf.get("mask"))
        roi_h5 = np.array(hf.get("roi"))
        g2_h5 = np.array(hf.get("g2"))
        taus_h5 = np.array(hf.get("taus"))
        g2b_h5 = np.array(hf.get("g2b"))
        taus2_h5 = np.array(hf.get("taus2"))
        if "g12b" in hf:
            g12b_h5 = np.array(hf.get("g12b"))

    if "g12b" in hf:
        return (
            meta_data,
            avg_h5,
            mask_h5,
            roi_h5,
            g2_h5,
            taus_h5,
            g2b_h5,
            taus2_h5,
            g12b,
        )
    else:
        return meta_data, avg_h5, mask_h5, roi_h5, g2_h5, taus_h5, g2b_h5, taus2_h5


def make_pdf_report(
    data_dir,
    uid,
    pdf_out_dir,
    pdf_filename,
    username,
    run_fit_form,
    run_one_time,
    run_two_time,
    run_four_time,
    run_xsvs,
    run_dose=None,
    oavs_report=False,
    report_type="saxs",
    md=None,
    report_invariant=False,
    return_class=False,
    res_h5_filename=None,
):

    if uid.startswith("uid=") or uid.startswith("Uid="):
        uid = uid[4:]
    c = create_pdf_report(
        data_dir,
        uid,
        pdf_out_dir,
        filename=pdf_filename,
        user=username,
        report_type=report_type,
        md=md,
        res_h5_filename=res_h5_filename,
    )
    # print( c.md)
    # Page one: Meta-data/Iq-Q/ROI
    c.report_header(page=1)
    c.report_meta(top=730)
    c.report_static(top=540, iq_fit=run_fit_form)
    c.report_ROI(top=290)
    page = 1
    ##Page Two for plot OVAS images if oavs_report is True
    if oavs_report:
        c.new_page()
        c.report_header(page=2)
        c.report_oavs(top=720, oavs_file=None, new_page=True)
        page += 1

    # Page Two: img~t/iq~t/waterfall/mean~t/g2/rate~q
    c.new_page()
    page += 1
    c.report_header(page=page)

    if c.report_type != "ang_saxs":
        c.report_time_analysis(top=720)
        if run_one_time:
            if c.report_type != "ang_saxs":
                top = 350
            else:
                top = 500
            if c.g2_fit_new_page:
                c.new_page()
                page += 1
                top = 720
            c.report_one_time(top=top)

        # self.two_g2_new_page = True
        # self.g2_fit_new_page = True

        # Page Three: two-time/two g2

        if run_two_time:
            c.new_page()
            page += 1
            c.report_header(page=page)
            c.report_two_time(top=720)

        if run_four_time:
            c.new_page()
            page += 1
            c.report_header(page=page)
            c.report_four_time(top=720)

        if run_xsvs:
            c.new_page()
            page += 1
            c.report_header(page=page)
            c.report_xsvs(top=720)
        if run_dose:
            c.new_page()
            page += 1
            c.report_header(page=page)
            c.report_dose(top=702)
        if report_invariant:
            c.new_page()
            page += 1
            c.report_header(page=page)
            c.report_invariant(top=702)

    else:
        c.report_flow_pv_g2(top=720, new_page=True)
        c.report_flow_pv_two_time(top=720, new_page=True)

    c.save_page()
    c.done()
    if return_class:
        return c


######################################
###Deal with saving dict to hdf5 file
def save_dict_to_hdf5(dic, filename):
    """
    ....
    """
    with h5py.File(filename, "w") as h5file:
        recursively_save_dict_contents_to_group(h5file, "/", dic)


def load_dict_from_hdf5(filename):
    """
    ....
    """
    with h5py.File(filename, "r") as h5file:
        return recursively_load_dict_contents_from_group(h5file, "/")


def recursively_save_dict_contents_to_group(h5file, path, dic):
    """..."""
    # argument type checking
    if not isinstance(dic, dict):
        raise ValueError("must provide a dictionary")

    if not isinstance(path, str):
        raise ValueError("path must be a string")
    if not isinstance(h5file, h5py._hl.files.File):
        raise ValueError("must be an open h5py file")
    # save items to the hdf5 file
    for key, item in dic.items():
        # print(key,item)
        key = str(key)
        if isinstance(item, list):
            item = np.array(item)
            # print(item)
        if not isinstance(key, str):
            raise ValueError("dict keys must be strings to save to hdf5")
        # save strings, numpy.int64, and numpy.float64 types
        if isinstance(
            item, (np.int64, np.float64, str, np.float, float, np.float32, int)
        ):
            # print( 'here' )
            h5file[path + key] = item
            if not h5file[path + key].value == item:
                raise ValueError(
                    "The data representation in the HDF5 file does not match the original dict."
                )
        # save numpy arrays
        elif isinstance(item, np.ndarray):
            try:
                h5file[path + key] = item
            except:
                item = np.array(item).astype("|S9")
                h5file[path + key] = item
            if not np.array_equal(h5file[path + key].value, item):
                raise ValueError(
                    "The data representation in the HDF5 file does not match the original dict."
                )
        # save dictionaries
        elif isinstance(item, dict):
            recursively_save_dict_contents_to_group(h5file, path + key + "/", item)
        # other types cannot be saved and will result in an error
        else:
            # print(item)
            raise ValueError("Cannot save %s type." % type(item))


def recursively_load_dict_contents_from_group(h5file, path):
    """..."""
    ans = {}
    for key, item in h5file[path].items():
        if isinstance(item, h5py._hl.dataset.Dataset):
            ans[key] = item.value
        elif isinstance(item, h5py._hl.group.Group):
            ans[key] = recursively_load_dict_contents_from_group(
                h5file, path + key + "/"
            )
    return ans


def export_xpcs_results_to_h5(filename, export_dir, export_dict):
    """
    YG. May 10, 2017
    save the results to a h5 file

    YG. Aug28 2019 modify, add try in export pandas to h5 to fit the new version of pandas

    filename:  the h5 file name
    export_dir: the exported file folder
    export_dict: dict, with keys as md, g2, g4 et.al.
    """

    fout = export_dir + filename
    dicts = ["md", "qval_dict", "qval_dict_v", "qval_dict_p"]
    dict_nest = ["taus_uids", "g2_uids"]

    with h5py.File(fout, "w") as hf:
        flag = False
        for key in list(export_dict.keys()):
            # print( key )
            if key in dicts:  # =='md' or key == 'qval_dict':
                md = export_dict[key]
                meta_data = hf.create_dataset(key, (1,), dtype="i")
                for key_ in md.keys():
                    try:
                        meta_data.attrs[str(key_)] = md[key_]
                    except:
                        pass
            elif key in dict_nest:
                # print(key)
                try:
                    recursively_save_dict_contents_to_group(
                        hf, "/%s/" % key, export_dict[key]
                    )
                except:
                    print("Can't export the key: %s in this dataset." % key)

            elif key in [
                "g2_fit_paras",
                "g2b_fit_paras",
                "spec_km_pds",
                "spec_pds",
                "qr_1d_pds",
            ]:
                try:
                    export_dict[key].to_hdf(
                        fout,
                        key=key,
                        mode="a",
                    )
                except:
                    flag = True
            else:
                data = hf.create_dataset(key, data=export_dict[key])
                # add this fill line at Octo 27, 2017
                data.set_fill_value = np.nan
    if flag:
        for key in list(export_dict.keys()):
            if key in [
                "g2_fit_paras",
                "g2b_fit_paras",
                "spec_km_pds",
                "spec_pds",
                "qr_1d_pds",
            ]:
                export_dict[key].to_hdf(
                    fout,
                    key=key,
                    mode="a",
                )

    print(
        "The xpcs analysis results are exported to %s with filename as %s"
        % (export_dir, filename)
    )


def extract_xpcs_results_from_h5_debug(
    filename, import_dir, onekey=None, exclude_keys=None
):
    """
    YG. Dec 22, 2016
    extract data from a h5 file

    filename:  the h5 file name
    import_dir: the imported file folder
    onekey: string, if not None, only extract that key
    return:
        extact_dict: dict, with keys as md, g2, g4 et.al.
    """

    import pandas as pds
    import numpy as np

    extract_dict = {}
    fp = import_dir + filename
    pds_type_keys = []
    dicts = ["md", "qval_dict", "qval_dict_v", "qval_dict_p", "taus_uids", "g2_uids"]
    if exclude_keys is None:
        exclude_keys = []
    if onekey is None:
        for k in dicts:
            extract_dict[k] = {}
        with h5py.File(fp, "r") as hf:
            # print (list( hf.keys()) )
            for key in list(hf.keys()):
                if key not in exclude_keys:
                    if key in dicts:
                        extract_dict[key] = recursively_load_dict_contents_from_group(
                            hf, "/" + key + "/"
                        )
                    elif key in [
                        "g2_fit_paras",
                        "g2b_fit_paras",
                        "spec_km_pds",
                        "spec_pds",
                        "qr_1d_pds",
                    ]:
                        pds_type_keys.append(key)
                    else:
                        extract_dict[key] = np.array(hf.get(key))
        for key in pds_type_keys:
            if key not in exclude_keys:
                extract_dict[key] = pds.read_hdf(fp, key=key)
    else:
        if onekey == "md":
            with h5py.File(fp, "r") as hf:
                md = hf.get("md")
                for key in list(md.attrs):
                    extract_dict["md"][key] = md.attrs[key]
        elif onekey in [
            "g2_fit_paras",
            "g2b_fit_paras",
            "spec_km_pds",
            "spec_pds",
            "qr_1d_pds",
        ]:
            extract_dict[onekey] = pds.read_hdf(fp, key=onekey)
        else:
            try:
                with h5py.File(fp, "r") as hf:
                    extract_dict[onekey] = np.array(hf.get(onekey))
            except:
                print("The %s dosen't have this %s value" % (fp, onekey))
    return extract_dict


def export_xpcs_results_to_h5_old(filename, export_dir, export_dict):
    """
    YG. Dec 22, 2016
    save the results to a h5 file

    filename:  the h5 file name
    export_dir: the exported file folder
    export_dict: dict, with keys as md, g2, g4 et.al.
    """
    import h5py

    fout = export_dir + filename
    dicts = ["md", "qval_dict", "qval_dict_v", "qval_dict_p"]  # {k1: { }}
    dict_nest = ["taus_uids", "g2_uids"]  # {k1: {k2:}}
    with h5py.File(fout, "w") as hf:
        for key in list(export_dict.keys()):
            # print( key )
            if key in dicts:  # =='md' or key == 'qval_dict':
                md = export_dict[key]
                meta_data = hf.create_dataset(key, (1,), dtype="i")
                for key_ in md.keys():
                    try:
                        meta_data.attrs[str(key_)] = md[key_]
                    except:
                        pass
            elif key in dict_nest:
                k1 = export_dict[key]
                v1 = hf.create_dataset(key, (1,), dtype="i")
                for k2 in k1.keys():

                    v2 = hf.create_dataset(k1, (1,), dtype="i")

            elif key in [
                "g2_fit_paras",
                "g2b_fit_paras",
                "spec_km_pds",
                "spec_pds",
                "qr_1d_pds",
            ]:
                export_dict[key].to_hdf(
                    fout,
                    key=key,
                    mode="a",
                )
            else:
                data = hf.create_dataset(key, data=export_dict[key])
    print(
        "The xpcs analysis results are exported to %s with filename as %s"
        % (export_dir, filename)
    )


def extract_xpcs_results_from_h5(
    filename, import_dir, onekey=None, exclude_keys=None, two_time_qindex=None
):
    """
    YG. Dec 22, 2016
    extract data from a h5 file

    filename:  the h5 file name
    import_dir: the imported file folder
    onekey: string, if not None, only extract that key
    return:
        extact_dict: dict, with keys as md, g2, g4 et.al.
    """

    import pandas as pds
    import numpy as np

    extract_dict = {}
    fp = import_dir + filename
    pds_type_keys = []
    dicts = ["md", "qval_dict", "qval_dict_v", "qval_dict_p", "taus_uids", "g2_uids"]
    if exclude_keys is None:
        exclude_keys = []
    if onekey is None:
        for k in dicts:
            extract_dict[k] = {}
        with h5py.File(fp, "r") as hf:
            # print (list( hf.keys()) )
            for key in list(hf.keys()):
                if key not in exclude_keys:
                    if key in dicts:
                        md = hf.get(key)
                        for key_ in list(md.attrs):
                            # print(key, key_)
                            if key == "qval_dict":
                                extract_dict[key][int(key_)] = md.attrs[key_]
                            else:
                                extract_dict[key][key_] = md.attrs[key_]

                    elif key in [
                        "g2_fit_paras",
                        "g2b_fit_paras",
                        "spec_km_pds",
                        "spec_pds",
                        "qr_1d_pds",
                    ]:
                        pds_type_keys.append(key)
                    else:
                        if key == "g12b":
                            if two_time_qindex is not None:
                                extract_dict[key] = hf.get(key)[:, :, two_time_qindex]
                            else:
                                extract_dict[key] = hf.get(key)[:]
                        else:
                            extract_dict[key] = hf.get(key)[
                                :
                            ]  # np.array( hf.get( key  ))

        for key in pds_type_keys:
            if key not in exclude_keys:
                extract_dict[key] = pds.read_hdf(fp, key=key)
    else:
        if onekey == "md":
            with h5py.File(fp, "r") as hf:
                md = hf.get("md")
                for key in list(md.attrs):
                    extract_dict["md"][key] = md.attrs[key]
        elif onekey in [
            "g2_fit_paras",
            "g2b_fit_paras",
            "spec_km_pds",
            "spec_pds",
            "qr_1d_pds",
        ]:
            extract_dict[onekey] = pds.read_hdf(fp, key=onekey)
        else:
            try:
                with h5py.File(fp, "r") as hf:
                    if key == "g12b":
                        if two_time_qindex is not None:
                            extract_dict[key] = hf.get(key)[:, :, two_time_qindex]
                        else:
                            extract_dict[key] = hf.get(key)[:]
                    else:
                        extract_dict[key] = hf.get(key)[:]  # np.array( hf.get( key  ))
                    # extract_dict[onekey] = hf.get( key  )[:] #np.array( hf.get( onekey  ))
            except:
                print("The %s dosen't have this %s value" % (fp, onekey))
    return extract_dict


def read_contrast_from_multi_csv(uids, path, times=None, unit=20):
    """Y.G. 2016, Dec 23, load contrast from multi csv file"""

    N = len(uids)
    if times is None:
        times = np.array([0] + [2 ** i for i in range(N)]) * unit
    for i, uid in enumerate(uids):
        fp = path + uid + "/uid=%s--contrast_factorL.csv" % uid
        contri = pds.read_csv(fp)
        qs = np.array(contri[contri.columns[0]])
        contri_ = np.array(contri[contri.columns[1]])
        if i == 0:
            contr = np.zeros([N, len(qs)])
        contr[i] = contri_
        # contr[0,:] = np.nan
    return times, contr


def read_contrast_from_multi_h5(
    uids,
    path,
):
    """Y.G. 2016, Dec 23, load contrast from multi h5 file"""
    N = len(uids)
    times_xsvs = np.zeros(N)
    for i, uid in enumerate(uids):
        t = extract_xpcs_results_from_h5(
            filename="%s_Res.h5" % uid, import_dir=path + uid + "/", onekey="times_xsvs"
        )
        times_xsvs[i] = t["times_xsvs"][0]
        contri = extract_xpcs_results_from_h5(
            filename="%s_Res.h5" % uid,
            import_dir=path + uid + "/",
            onekey="contrast_factorL",
        )
        if i == 0:
            contr = np.zeros([N, contri["contrast_factorL"].shape[0]])
        contr[i] = contri["contrast_factorL"][:, 0]
    return times_xsvs, contr
