"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
function testStack() {
    const stack = new cdk.Stack(undefined, undefined, { env: { account: '12345', region: 'us-test-1' } });
    stack.node.setContext('availability-zones:12345:us-test-1', ['us-test-1a', 'us-test-1b']);
    return stack;
}
module.exports = {
    'creating a Cluster also creates 2 DB Instances'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Properties: {
                Engine: 'aurora',
                DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
                MasterUsername: 'admin',
                MasterUserPassword: 'tooshort',
                VpcSecurityGroupIds: [{ 'Fn::GetAtt': ['DatabaseSecurityGroup5C91FDCB', 'GroupId'] }],
            },
            DeletionPolicy: assert_1.ABSENT,
            UpdateReplacePolicy: 'Snapshot',
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.countResources('AWS::RDS::DBInstance', 2));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DeletionPolicy: assert_1.ABSENT,
            UpdateReplacePolicy: assert_1.ABSENT,
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'can create a cluster with a single instance'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora',
            DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
            MasterUsername: 'admin',
            MasterUserPassword: 'tooshort',
            VpcSecurityGroupIds: [{ 'Fn::GetAtt': ['DatabaseSecurityGroup5C91FDCB', 'GroupId'] }],
        }));
        test.done();
    },
    'can create a cluster with imported vpc and security group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = ec2.Vpc.fromLookup(stack, 'VPC', {
            vpcId: 'VPC12345',
        });
        const sg = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'SecurityGroupId12345');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
                securityGroup: sg,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora',
            DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
            MasterUsername: 'admin',
            MasterUserPassword: 'tooshort',
            VpcSecurityGroupIds: ['SecurityGroupId12345'],
        }));
        test.done();
    },
    'cluster with parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const group = new lib_1.ClusterParameterGroup(stack, 'Params', {
            family: 'hello',
            description: 'bye',
            parameters: {
                param: 'value',
            },
        });
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            parameterGroup: group,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            DBClusterParameterGroupName: { Ref: 'ParamsA8366201' },
        }));
        test.done();
    },
    "sets the retention policy of the SubnetGroup to 'Retain' if the Cluster is created with 'Retain'"(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        new lib_1.DatabaseCluster(stack, 'Cluster', {
            masterUser: { username: 'admin' },
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE),
                vpc,
            },
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBSubnetGroup', {
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain',
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'creates a secret when master credentials are not specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            MasterUsername: {
                'Fn::Join': [
                    '',
                    [
                        '{{resolve:secretsmanager:',
                        {
                            Ref: 'DatabaseSecret3B817195',
                        },
                        ':SecretString:username::}}',
                    ],
                ],
            },
            MasterUserPassword: {
                'Fn::Join': [
                    '',
                    [
                        '{{resolve:secretsmanager:',
                        {
                            Ref: 'DatabaseSecret3B817195',
                        },
                        ':SecretString:password::}}',
                    ],
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                ExcludeCharacters: '\"@/\\',
                GenerateStringKey: 'password',
                PasswordLength: 30,
                SecretStringTemplate: '{"username":"admin"}',
            },
        }));
        test.done();
    },
    'create an encrypted cluster with custom KMS key'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            kmsKey: new kms.Key(stack, 'Key'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            KmsKeyId: {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'cluster with instance parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const parameterGroup = new lib_1.ParameterGroup(stack, 'ParameterGroup', {
            family: 'hello',
            parameters: {
                key: 'value',
            },
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                parameterGroup,
                vpc,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBParameterGroupName: {
                Ref: 'ParameterGroup5E32DECB',
            },
        }));
        test.done();
    },
    'create a cluster using a specific version of MySQL'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            engineVersion: '5.7.mysql_aurora.2.04.4',
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora-mysql',
            EngineVersion: '5.7.mysql_aurora.2.04.4',
        }));
        test.done();
    },
    'create a cluster using a specific version of Postgresql'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_POSTGRESQL,
            engineVersion: '10.7',
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora-postgresql',
            EngineVersion: '10.7',
        }));
        test.done();
    },
    'cluster exposes different read and write endpoints'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        test.notDeepEqual(stack.resolve(cluster.clusterEndpoint), stack.resolve(cluster.clusterReadEndpoint));
        test.done();
    },
    'imported cluster with imported security group honors allowAllOutbound'(test) {
        // GIVEN
        const stack = testStack();
        const cluster = lib_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        // WHEN
        cluster.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-123456789',
        }));
        test.done();
    },
    'cluster with enabled monitoring'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            monitoringInterval: cdk.Duration.minutes(1),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                'Fn::GetAtt': ['DatabaseMonitoringRole576991DA', 'Arn'],
            },
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'monitoring.rds.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            ManagedPolicyArns: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/service-role/AmazonRDSEnhancedMonitoringRole',
                        ],
                    ],
                },
            ],
        }));
        test.done();
    },
    'create a cluster with imported monitoring role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const monitoringRole = new aws_iam_1.Role(stack, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            monitoringInterval: cdk.Duration.minutes(1),
            monitoringRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                'Fn::GetAtt': ['MonitoringRole90457BF9', 'Arn'],
            },
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'throws when trying to add rotation to a cluster without secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        test.throws(() => cluster.addRotationSingleUser(), /without secret/);
        test.done();
    },
    'throws when trying to add single user rotation multiple times'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: { username: 'admin' },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // WHEN
        cluster.addRotationSingleUser();
        // THEN
        test.throws(() => cluster.addRotationSingleUser(), /A single user rotation was already added to this cluster/);
        test.done();
    },
    'create a cluster with s3 import role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const associatedRole = new aws_iam_1.Role(stack, 'AssociatedRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportRole: associatedRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'AssociatedRole824CFCD3',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'AssociatedRole824CFCD3',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 import buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'Bucket83908E77',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'Bucket83908E77',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'create a cluster with s3 export role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const associatedRole = new aws_iam_1.Role(stack, 'AssociatedRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportRole: associatedRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'AssociatedRole824CFCD3',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'AssociatedRole824CFCD3',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 export buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                            's3:DeleteObject*',
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'Bucket83908E77',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'Bucket83908E77',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'create a cluster with s3 import and export buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportBuckets: [importBucket],
            s3ExportBuckets: [exportBucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                },
                {
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 import and export buckets and custom parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const parameterGroup = new lib_1.ClusterParameterGroup(stack, 'ParameterGroup', {
            family: 'family',
            parameters: {
                key: 'value',
            },
        });
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            parameterGroup,
            s3ImportBuckets: [importBucket],
            s3ExportBuckets: [exportBucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                },
                {
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'family',
            Parameters: {
                key: 'value',
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'PostgreSQL cluster with s3 export buckets does not generate custom parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_POSTGRESQL,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup'));
        test.done();
    },
    'throws when s3ExportRole and s3ExportBuckets properties are both specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const exportRole = new aws_iam_1.Role(stack, 'ExportRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // THEN
        test.throws(() => new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportRole: exportRole,
            s3ExportBuckets: [exportBucket],
        }));
        test.done();
    },
    'throws when s3ImportRole and s3ImportBuckets properties are both specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const importRole = new aws_iam_1.Role(stack, 'ImportRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        // THEN
        test.throws(() => new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportRole: importRole,
            s3ImportBuckets: [importBucket],
        }));
        test.done();
    },
    'does not throw (but adds a node error) if a (dummy) VPC does not have sufficient subnets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = ec2.Vpc.fromLookup(stack, 'VPC', { isDefault: true });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
                vpcSubnets: {
                    subnetName: 'DefinitelyDoesNotExist',
                },
            },
        });
        // THEN
        const art = assert_1.SynthUtils.synthesize(stack);
        const meta = art.findMetadataByType('aws:cdk:error');
        test.equal(meta[0].data, 'Cluster requires at least 2 subnets, got 0');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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