"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const targets = require("@aws-cdk/aws-events-targets");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const rds = require("../lib");
const app = new cdk.App();
class DatabaseInstanceStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const vpc = new ec2.Vpc(this, 'VPC', { maxAzs: 2 });
        /// !show
        // Set open cursors with parameter group
        const parameterGroup = new rds.ParameterGroup(this, 'ParameterGroup', {
            family: 'oracle-se1-11.2',
            parameters: {
                open_cursors: '2500',
            },
        });
        /// Add XMLDB and OEM with option group
        const optionGroup = new rds.OptionGroup(this, 'OptionGroup', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            majorEngineVersion: '11.2',
            configurations: [
                {
                    name: 'XMLDB',
                },
                {
                    name: 'OEM',
                    port: 1158,
                    vpc,
                },
            ],
        });
        // Allow connections to OEM
        optionGroup.optionConnections.OEM.connections.allowDefaultPortFromAnyIpv4();
        // Database instance with production values
        const instance = new rds.DatabaseInstance(this, 'Instance', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            licenseModel: rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
            multiAz: true,
            storageType: rds.StorageType.IO1,
            masterUsername: 'syscdk',
            vpc,
            databaseName: 'ORCL',
            storageEncrypted: true,
            backupRetention: cdk.Duration.days(7),
            monitoringInterval: cdk.Duration.seconds(60),
            enablePerformanceInsights: true,
            cloudwatchLogsExports: [
                'trace',
                'audit',
                'alert',
                'listener',
            ],
            cloudwatchLogsRetention: logs.RetentionDays.ONE_MONTH,
            autoMinorVersionUpgrade: false,
            optionGroup,
            parameterGroup,
        });
        // Allow connections on default port from any IPV4
        instance.connections.allowDefaultPortFromAnyIpv4();
        // Rotate the master user password every 30 days
        instance.addRotationSingleUser();
        // Add alarm for high CPU
        new cloudwatch.Alarm(this, 'HighCPU', {
            metric: instance.metricCPUUtilization(),
            threshold: 90,
            evaluationPeriods: 1,
        });
        // Trigger Lambda function on instance availability events
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline('exports.handler = (event) => console.log(event);'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const availabilityRule = instance.onEvent('Availability', { target: new targets.LambdaFunction(fn) });
        availabilityRule.addEventPattern({
            detail: {
                EventCategories: [
                    'availability',
                ],
            },
        });
        /// !hide
    }
}
new DatabaseInstanceStack(app, 'aws-cdk-rds-instance');
app.synth();
//# sourceMappingURL=data:application/json;base64,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