"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const version_1 = require("./private/version");
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    // tslint:disable-next-line max-line-length
    constructor(name, singleUserRotationApplication, multiUserRotationApplication, parameterGroupFamilies) {
        this.name = name;
        this.singleUserRotationApplication = singleUserRotationApplication;
        this.multiUserRotationApplication = multiUserRotationApplication;
        this.parameterGroupFamilies = parameterGroupFamilies;
    }
    /**
     * Get the latest parameter group family for this engine. Latest is determined using semver on the engine major version.
     * When `engineVersion` is specified, return the parameter group family corresponding to that engine version.
     * Return undefined if no parameter group family is defined for this engine or for the requested `engineVersion`.
     */
    parameterGroupFamily(engineVersion) {
        if (this.parameterGroupFamilies === undefined) {
            return undefined;
        }
        if (engineVersion) {
            const family = this.parameterGroupFamilies.find(x => engineVersion.startsWith(x.engineMajorVersion));
            if (family) {
                return family.parameterGroupFamily;
            }
        }
        else if (this.parameterGroupFamilies.length > 0) {
            const sorted = this.parameterGroupFamilies.slice().sort((a, b) => {
                return version_1.compare(a.engineMajorVersion, b.engineMajorVersion);
            }).reverse();
            return sorted[0].parameterGroupFamily;
        }
        return undefined;
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/* tslint:disable max-line-length */
DatabaseClusterEngine.AURORA = new DatabaseClusterEngine('aurora', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'aurora5.6' },
]);
DatabaseClusterEngine.AURORA_MYSQL = new DatabaseClusterEngine('aurora-mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.7', parameterGroupFamily: 'aurora-mysql5.7' },
]);
DatabaseClusterEngine.AURORA_POSTGRESQL = new DatabaseClusterEngine('aurora-postgresql', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.6', parameterGroupFamily: 'aurora-postgresql9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'aurora-postgresql10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'aurora-postgresql11' },
]);
//# sourceMappingURL=data:application/json;base64,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